package geosearch

import (
	"errors"
	"sort"
	"strings"

	"a.yandex-team.ru/travel/rasp/wizards/go/wizard_proxy_api/internal/application"
	"a.yandex-team.ru/travel/rasp/wizards/go/wizard_proxy_api/internal/geomodel"
	"a.yandex-team.ru/travel/rasp/wizards/go/wizard_proxy_api/internal/storage"
)

type PointSearcher interface {
	PointSearch(string, geomodel.TransportType) ([]geomodel.Point, map[int]bool, error)
}

type BasePointSearch struct {
	Cfg     *application.Config
	Storage *storage.Storage
}

var StopWords = map[string]bool{
	"пункт":   true,
	"ост":     true,
	"км":      true,
	"на":      true,
	"поворот": true,
	"трасса":  true,
	"сан":     true,
	"пасс":    true,
	"им":      true,
	"эль":     true,
	"порт":    true,
	"форт":    true,
	"до":      true,
	"де":      true,
}

func checkInStopWords(word string) bool {
	term := strings.Trim(word, " -.()")
	_, found := StopWords[term]
	return found
}

func normalize(title string) string { // TODO: dummy, from common.utils.text
	return title
}

func cleanTitle(title string) string { // TODO: dummy
	return title
}

func titleVariants(title string) []string { // TODO: dummy
	return []string{title}
}

func isDefaultTitlePoint(title string, point geomodel.Point) bool { // TODO: dummy, move to geosearch.models, add repo
	// geosearch.models.DefaultPoint.isDefaultTitlePoint
	return true
}

func findBySlug(value string) (geomodel.Point, error) { // TODO: dummy. from common.utils.point_slugs
	return nil, errors.New("point was not found")
}

func findStationsFiltered(s *storage.Storage, dict storage.NameSearchIndexDict, title string, filterFunc func(*storage.Storage, *geomodel.Station) bool) ([]*geomodel.Station, error) {
	stations, err := s.NameSearcher.FindStations(dict, title)
	if err != nil {
		return nil, err
	}
	var result []*geomodel.Station
	for _, p := range stations {
		if filterFunc(s, p) {
			result = append(result, p)
		}
	}
	return result, nil
}

func findSettlementsFiltered(s *storage.Storage, dict storage.NameSearchIndexDict, title string, filterFunc func(*geomodel.Settlement) bool) ([]*geomodel.Settlement, error) {
	settlements, err := s.NameSearcher.FindSettlements(dict, title)
	if err != nil {
		return nil, err
	}
	var result []*geomodel.Settlement
	for _, p := range settlements {
		if filterFunc(p) {
			result = append(result, p)
		}
	}
	return result, nil
}

// Сортирует города по (majority, id)
func sortedSettlements(settlements []*geomodel.Settlement) []*geomodel.Settlement {
	// TODO: may be sort with precomputed keys? sort by id and stable sort by majority
	sort.Slice(settlements, func(i, j int) bool {
		iID := settlements[i].GetID()
		jID := settlements[j].GetID()
		iMajority := settlements[i].GetMajorityByteValue()
		jMajority := settlements[j].GetMajorityByteValue()
		return (iMajority < jMajority) || (iMajority == jMajority && iID < jID)
	})
	return settlements
}

// Сортирует станции по (majority, title)
func sortedStations(stations []*geomodel.Station) []*geomodel.Station {
	// TODO: may be sort with precompute keys?
	sort.Slice(stations, func(i, j int) bool {
		iTitle := stations[i].GetTitle()
		jTitle := stations[j].GetTitle()
		iMajority := stations[i].GetMajorityByteValue()
		jMajority := stations[j].GetMajorityByteValue()
		return (iMajority < jMajority) || (iMajority == jMajority && iTitle < jTitle)
	})
	return stations
}

func FindAllPoints(pointSearch PointSearcher, title string, tType geomodel.TransportType) ([]geomodel.Point, map[int]bool, error) {
	clean := cleanTitle(title)
	if checkInStopWords(clean) {
		return nil, nil, errors.New("title is in stop words")
	}
	if len(clean) < 2 {
		return nil, nil, errors.New("title is too short")
	}
	for _, v := range titleVariants(clean) {
		points, ids, err := pointSearch.PointSearch(v, tType)
		if err == nil {
			return points, ids, nil
		}
	}
	return nil, nil, errors.New("points were not found")
}

func GetPointByKey(pointKey string) (geomodel.Point, error) { // TODO: dummy, move to common/geomodel
	return nil, errors.New("point was not found by point key")
}

func FindPointList(pointSearch PointSearcher, title string, tType geomodel.TransportType, pointKey string, slug string) (*PointList, error) {
	var variants []geomodel.Point
	var seenIds map[int]bool
	var point geomodel.Point
	var err error
	if slug != "" && pointKey != "" {
		return nil, errors.New("excess point key")
	}
	if slug != "" {
		point, err = findBySlug(slug)
		if err != nil || point.IsHidden() {
			return nil, errors.New("invalid slug")
		}
	}
	if pointKey != "" {
		point, err = GetPointByKey(pointKey)
		if err != nil {
			return nil, err
		}
	}
	if title != "" {
		variants, seenIds, err = FindAllPoints(pointSearch, title, tType)
		if err != nil {
			return nil, err
		}
	}
	if point != nil {
		_, found := seenIds[point.GetID()]
		if !found {
			variants = append(variants, point) // TODO as in original, but may be append to start?
		}
	}
	if len(variants) == 0 {
		return nil, errors.New("points were not wound")
	}
	return NewPointListWithVariants(point, variants, title, point != nil)
}
