package geosearch

import (
	"errors"
	"strings"

	"a.yandex-team.ru/travel/rasp/wizards/go/wizard_proxy_api/internal/geomodel"
	"a.yandex-team.ru/travel/rasp/wizards/go/wizard_proxy_api/internal/storage"
)

type StationReplacement struct {
	dir string
	id  int
}

// Замена города на станцию. RASP-2338
var PointReplacements = map[string][]StationReplacement{
	"москва": []StationReplacement{
		StationReplacement{dir: "msk_bel", id: 2000006},  // Белорусский вокзал
		StationReplacement{dir: "msk_gor", id: 2000001},  // Курский вокзал
		StationReplacement{dir: "msk_kaz", id: 2000003},  // Казанский вокзал
		StationReplacement{dir: "msk_kiv", id: 2000007},  // Киевский вокзал
		StationReplacement{dir: "msk_kur", id: 2000001},  // Курский вокзал
		StationReplacement{dir: "msk_len", id: 2006004},  // Ленинградский вокзал
		StationReplacement{dir: "msk_pav", id: 2000005},  // Павелецкий вокзал
		StationReplacement{dir: "msk_sav", id: 2000009},  // Савёловский вокзал
		StationReplacement{dir: "msk_yar", id: 2000002},  // Ярославский вокзал
		StationReplacement{dir: "msk_riz", id: 2000008},  // Рижский вокзал
		StationReplacement{dir: "msk_mkzd", id: 9601334}, // Локомотив

	},
	"санкт-петербург": []StationReplacement{
		StationReplacement{dir: "spb_vit", id: 9602496}, // Витебский вокзал
		StationReplacement{dir: "spb_vyb", id: 9602497}, // Финляндский вокзал
		StationReplacement{dir: "spb_msk", id: 9602494}, // Московский вокзал
		StationReplacement{dir: "spb_pri", id: 9602497}, // Финляндский вокзал
		StationReplacement{dir: "spb_lad", id: 9602497}, // Финляндский вокзал
		StationReplacement{dir: "spb_bal", id: 9602498}, // Балтийский вокзал
	},
}

// Эти города не надо заменять на станции
var PreservedCitiesIDs = map[int]bool{
	213: true, // Москва
	2:   true, // Санкт-Петербург
	11:  true, // Рязань
	15:  true, // Тула
	147: true, // Харьков
	157: true, // Минск
}

// Обработка полученных списков городов
func ProcessPointsLists(s *storage.Storage, departurePointList *PointList, arrivalPointList *PointList, clientCity *geomodel.Settlement, disableReplace bool, suburban bool, disableReduceFrom bool, disableReduceTo bool) (*PointList, *PointList, error) {
	// TODO make logging

	// RASP-3259: Выбирать из множества одноименных станций те, что в регионе пользователя
	departurePointList, arrivalPointList = choozeStationsInUserRegion(departurePointList, arrivalPointList, clientCity)

	// RASP-4171, RASP-6631: Выбираем станции одного внешнего направления
	departurePointList, arrivalPointList = chooseStationsForOneDirection(departurePointList, arrivalPointList)

	// RASP-3435: Выбираем станции в одной пригородной зоне
	departurePointList, arrivalPointList = chooseStationsForOneSuburbZone(departurePointList, arrivalPointList)

	// RASP-10218 Выбрать города на одном внешнем направлении
	departurePointList, arrivalPointList = chooseSettlementsForOneDirection(suburban, departurePointList, arrivalPointList)

	// Если города до сих пор не выбраны, то берем первый вариант
	departurePointList.choozeFirstIfNotChoosed()
	arrivalPointList.choozeFirstIfNotChoosed()

	// Выбираем первый вариант, не равный другому выбранному
	departurePointList, arrivalPointList, err := choozeFirstVariantNotEqualToAnotherChoosed(departurePointList, arrivalPointList)
	if err != nil {
		return nil, nil, err
	}

	// Убираем из вариантов другого конца точно указанные пункты
	if arrivalPointList.IsExactVariant {
		// TODO filter
		departurePointList = removeExactVariants(departurePointList)
	}

	if departurePointList.IsExactVariant {
		// TODO filter
		arrivalPointList = removeExactVariants(arrivalPointList)
	}

	// Колдунство замены города на станцию
	departurePointList, arrivalPointList = changeSettlementToStation(s, departurePointList, arrivalPointList, disableReplace, suburban, disableReduceFrom, disableReduceTo)

	// RASP-9636 при поиске автобуса внутри города требовать уточнения
	// TODO

	// RASP-4084: сужение города до станции
	// RASP-11756: отключаем сужение
	if !disableReduceFrom {
		departurePointList, _ = reduceCity(arrivalPointList.MainPoint, departurePointList, suburban)
	}
	// RASP-11756: отключаем сужение
	if !disableReduceTo {
		arrivalPointList, _ = reduceCity(departurePointList.MainPoint, arrivalPointList, suburban)
	}

	if departurePointList.MainPoint == arrivalPointList.MainPoint { // TODO: check != nil ?
		return nil, nil, errors.New("found same points for departure and arrival")
	}

	return departurePointList, arrivalPointList, nil
}

// RASP-3375, RASP-4084 - сужение города до станции
func reduceCity(point geomodel.Point, pointList *PointList, suburban bool) (*PointList, error) { // TODO
	if point.GetType() != geomodel.EStation {
		return pointList, errors.New("point is not a station") // TODO нам точно нужен возврат ошибки в этой функции?
	}
	station := point
	pl, _ := NewPointList(station, false)
	return pl, nil
}

// Колдунство замены города на станцию
func changeSettlementToStation(s *storage.Storage, departurePointList *PointList, arrivalPointList *PointList, disableReplace bool, suburban bool, disableReduceFrom bool, disableReduceTo bool) (*PointList, *PointList) {
	if !disableReplace && (allowReplace(departurePointList.MainPoint, arrivalPointList.MainPoint, suburban) ||
		allowReplace(arrivalPointList.MainPoint, departurePointList.MainPoint, suburban)) {

		// RASP-11756
		if !disableReduceFrom {
			departurePointList = replacePoint(s, departurePointList, arrivalPointList)
		}
		// RASP-11756
		if !disableReduceTo {
			arrivalPointList = replacePoint(s, arrivalPointList, departurePointList)
		}
	}
	return departurePointList, arrivalPointList
}

// Определяет, можно ли применять колдунство по замене города на станцию
// TODO может стоит поменять логику
func allowReplace(departure geomodel.Point, arrival geomodel.Point, suburban bool) bool { // TODO dummy
	return false
}

// Запрещать колдунство, если обе станции принадлежат разным главным городам направлений. RASP-4467
func allowReplaceForMainCities(departure geomodel.Point, arrival geomodel.Point) bool { // TODO dummy
	return false
}

// Колдунство. Замена города на станцию. RASP-2338
func replacePoint(s *storage.Storage, pointList1 *PointList, pointList2 *PointList) *PointList { // TODO dummy
	stations := []*geomodel.Station{}
	var found bool

	if pointList2.MainPoint.GetType() == geomodel.EStation {
		station, err := s.GetStationByID(pointList2.MainPoint.GetID(), false) // TODO find better way
		if err != nil {
			return pointList1
		}
		stations = append(stations, station)
	} else if pointList2.MainPoint.GetType() == geomodel.ESettlement {
		stations, found = s.GetStationsBySettlement(pointList2.MainPoint)
		if !found {
			return pointList1
		}
	} else {
		return pointList1
	}

	// взять множество направлений, соответствующих станциям
	// сделать захардкоженную замену, отдать поинтлист по этой станции
	dirs, err := s.GetExternalDirectionsForStations(stations)
	if err != nil {
		return pointList1
	}
	dirsSet := map[string]bool{}
	for _, id := range dirs {
		dirsSet[id] = true
	}

	replacements, found := PointReplacements[strings.ToLower(pointList1.MainPoint.GetTitle())] // TODO can we change key to id?
	if !found {
		return pointList1
	}

	for _, rep := range replacements {
		_, found := dirsSet[rep.dir] // TODO: can we use direction_id here?
		if !found {
			continue
		}
		station, err := s.GetStationByID(rep.id, false) // TODO: check pk and id is the same things
		if err != nil {
			continue // TODO do we need log it? this station must not make errors
		}
		pointlist, err := NewPointListWithVariants(station, []geomodel.Point{geomodel.Point(station)}, pointList1.Term, false)
		if err != nil {
			return pointList1 // TODO: I am not sure about this case
		}
		return pointlist
	}

	return pointList1
}

func needToReduce(station *geomodel.Station, pointList *PointList, suburban bool) bool {
	if !pointList.AllowReduce {
		return false
	}

	point := pointList.MainPoint
	if point.GetType() != geomodel.ESettlement {
		return false
	}

	// Стоп-лист
	pointID := point.GetID()
	_, found := PreservedCitiesIDs[pointID]
	if found {
		return false
	}

	if suburban {
		return true
	}

	if station.GetType() != geomodel.EStation {
		return false
	}

	//  Аэропорты не сужаем
	if station.GetTransportType() == geomodel.ETransportPlane {
		return false
	}

	stationID, err := station.GetSettlementID()
	if err != nil {
		return false
	}
	return stationID == pointID
}

func removeExactVariants(pointList *PointList) *PointList { // TODO dummy
	return pointList
}

// RASP-4171, RASP-6631: Выбираем станции одного внешнего направления
func chooseStationsForOneDirection(departurePointList *PointList, arrivalPointList *PointList) (*PointList, *PointList) { // TODO dummy
	if (departurePointList.MainPoint == nil || arrivalPointList.MainPoint == nil) &&
		len(departurePointList.Variants) > 0 && len(arrivalPointList.Variants) > 0 {
		//stations_from = [v for v in point_list_from.variants if isinstance(v, Station)]
		//stations_to = [v for v in point_list_to.variants if isinstance(v, Station)]
		// TODO
		return departurePointList, arrivalPointList
	}
	return departurePointList, arrivalPointList
}

// RASP-3435: Выбираем станции в одной пригородной зоне
func chooseStationsForOneSuburbZone(departurePointList *PointList, arrivalPointList *PointList) (*PointList, *PointList) { // TODO dummy
	if (departurePointList.MainPoint == nil || arrivalPointList.MainPoint == nil) &&
		departurePointList.HasVariants() && arrivalPointList.HasVariants() {
		// TODO
		return departurePointList, arrivalPointList
	}
	return departurePointList, arrivalPointList
}

// RASP-10218 Выбрать города на одном внешнем направлении
func chooseSettlementsForOneDirection(suburban bool, departurePointList *PointList, arrivalPointList *PointList) (*PointList, *PointList) { // TODO dummy
	if suburban && (departurePointList.MainPoint == nil || arrivalPointList.MainPoint == nil) &&
		departurePointList.HasVariants() && arrivalPointList.HasVariants() {
		// TODO
		return departurePointList, arrivalPointList
	}
	return departurePointList, arrivalPointList
}

// RASP-3259: Выбирать из множества одноименных станций те, что в регионе пользователя
func choozeStationsInUserRegion(departurePointList *PointList, arrivalPointList *PointList, clientCity *geomodel.Settlement) (*PointList, *PointList) {
	if clientCity != nil && len(departurePointList.Variants) > 0 && len(arrivalPointList.Variants) > 0 {
		clientRegionID, err := clientCity.GetRegionID()
		if err == nil {
			if departurePointList.MainPoint == nil {
				for _, point := range departurePointList.Variants {
					regionID, err := point.GetRegionID()
					if err == nil && regionID == clientRegionID {
						departurePointList.MainPoint = point
						break
					}
				}
			}
			if arrivalPointList.MainPoint == nil {
				for _, point := range arrivalPointList.Variants {
					regionID, err := point.GetRegionID()
					if err == nil && regionID == clientRegionID {
						arrivalPointList.MainPoint = point
						break
					}
				}
			}
		}
	}
	return departurePointList, arrivalPointList
}

// Выбираем первый вариант, не равный другому выбранному
func choozeFirstVariantNotEqualToAnotherChoosed(departurePointList *PointList, arrivalPointList *PointList) (*PointList, *PointList, error) {
	if departurePointList.MainPoint == arrivalPointList.MainPoint {
		if arrivalPointList.HasVariants() {
			for _, point := range arrivalPointList.Variants {
				if point != departurePointList.MainPoint {
					arrivalPointList.MainPoint = point
					break
				}
			}
		} else if departurePointList.HasVariants() {
			for _, point := range departurePointList.Variants {
				if point != arrivalPointList.MainPoint {
					departurePointList.MainPoint = point
					break
				}
			}
		} else {
			return nil, nil, errors.New("found same points for departure and arrival")
		}
	}
	return departurePointList, arrivalPointList, nil
}
