package geosearch

import (
	"a.yandex-team.ru/travel/rasp/wizards/go/wizard_proxy_api/internal/application"
	"a.yandex-team.ru/travel/rasp/wizards/go/wizard_proxy_api/internal/geomodel"
	"a.yandex-team.ru/travel/rasp/wizards/go/wizard_proxy_api/internal/storage"
)

type SuburbanPointSearch struct {
	Cfg     *application.Config
	Storage *storage.Storage
}

var FirstPriorityAeroports = []int{
	9600215, // Внуково
	9600216, // Домодедово
}

var SuitableStationTypes = map[geomodel.TransportType]bool{
	geomodel.ETransportTrain: true,
	geomodel.ETransportPlane: true,
}

// Проверяет станцию, является ли она подходящей электричечной станцией
func isSuburbanSuitableStation(store *storage.Storage, station *geomodel.Station) bool {
	tType := station.GetTransportType()
	_, found := SuitableStationTypes[tType]
	if !found {
		return false
	}
	// Недостаточно важные
	majority := station.GetMajorityID()
	if majority > geomodel.StationMajorityNotInSearchID {
		return false
	}
	// Скрытые
	if station.IsHidden() {
		return false
	}

	if tType == geomodel.ETransportPlane {
		if !station.HasAeroexpress() {
			return false
		}
		// RASP-12633 В случае если название станции совпадает с названием города, брать станцию
		// (правило Коломны) - кроме тех случаев, когда эта станция имеет вид транспорта "Самолет"
		// TODO проверить тикет, комментарий отличается от кода, мб код неправ
		id, err := station.GetSettlementID()
		if err == nil {
			settlement, err := store.GetSettlementByID(id, false)
			if err == nil && station.GetTitle() == settlement.GetTitle() {
				return false
			}
		}
	}
	return true
}

// Является ли город подходящим, учитывая найденные станции
// не должно быть станции с таким же title
func getSuburbanSettlementFilter(store *storage.Storage, stations []*geomodel.Station) func(*geomodel.Settlement) bool {
	stationsTitles := make(map[int]string, len(stations))
	for _, s := range stations {
		id, err := s.GetSettlementID()
		if err != nil {
			continue
		}
		settlement, err := store.GetSettlementByID(id, false)
		if err == nil {
			stationsTitles[id] = normalize(settlement.GetTitle())
		}
	}

	filterFunction := func(settlement *geomodel.Settlement) bool {
		if settlement.IsHidden() {
			return false
		}
		normalizedTitle := normalize(settlement.GetTitle())
		stationTitle, found := stationsTitles[settlement.GetID()]
		if found && normalizedTitle == stationTitle {
			return false
		}
		return true
	}
	return filterFunction
}

// Ищет станции в NameSearch словарях, сортирует и дописывает в конец списка
func (pointSearch SuburbanPointSearch) AppendStations(title string, tType geomodel.TransportType, matchType storage.NameSearchIndexDict, points []geomodel.Point, seenIds map[int]bool) ([]geomodel.Point, map[int]bool, []*geomodel.Station) {

	stations, err := findStationsFiltered(pointSearch.Storage, matchType, title, isSuburbanSuitableStation)

	if err != nil {
		return points, seenIds, []*geomodel.Station{}
	}
	for _, p := range stations {
		seenIds[p.GetID()] = true
	}
	sortedStations := sortedStations(stations)
	for _, s := range sortedStations {
		points = append(points, geomodel.Point(s))
	}

	return points, seenIds, sortedStations
}

// Ищет города в NameSearch словарях, сортирует и дописывает в конец списка
func (pointSearch SuburbanPointSearch) AppendSettlements(title string, tType geomodel.TransportType, matchType storage.NameSearchIndexDict, stations []*geomodel.Station, points []geomodel.Point, seenIds map[int]bool) ([]geomodel.Point, map[int]bool) {

	suitableSettlement := getSuburbanSettlementFilter(pointSearch.Storage, stations)
	settlements, err := findSettlementsFiltered(pointSearch.Storage, matchType, title, suitableSettlement)
	if err != nil {
		return points, seenIds
	}
	for _, p := range settlements {
		seenIds[p.GetID()] = true
	}
	sortedSettlements := sortedSettlements(settlements)
	for _, s := range sortedSettlements {
		points = append(points, geomodel.Point(s))
	}
	return points, seenIds
}

// Возвращает отсортированный список найденных объектов, первый из блоков, который не пуст:
// сначала приоритетные аэропорты (RASP-2248, RASP-3375), потом станции, потом города // TODO (комментарий в оригинале не соответствует коду)
//
// Станции, найденные NameSearch Exact, отсортированные по (majority, title)
// Города, найденные NameSearch Exact, отсортированные по (majority, pointKey)
// либо
// Станции, найденные NameSearch Words, отсортированные по (majority, title)
// Города, найденные NameSearch Words, отсортированные по (majority, pointKey)
// либо
// Станци, полученные из CodeManager, по коду=title
func (pointSearch SuburbanPointSearch) PointSearch(title string, tType geomodel.TransportType) ([]geomodel.Point, map[int]bool, error) {
	seenIds := make(map[int]bool, pointSearch.Cfg.SearchPointLimit)
	result := make([]geomodel.Point, pointSearch.Cfg.SearchPointLimit)

	result, seenIds, stations := pointSearch.AppendStations(title, tType, storage.NameSearchExact, result, seenIds)
	result, seenIds = pointSearch.AppendSettlements(title, tType, storage.NameSearchExact, stations, result, seenIds)

	if len(result) > 0 {
		return result, seenIds, nil
	}

	result, seenIds, stations = pointSearch.AppendStations(title, tType, storage.NameSearchWords, result, seenIds)
	result, seenIds = pointSearch.AppendSettlements(title, tType, storage.NameSearchWords, stations, result, seenIds)

	if len(result) > 0 {
		return result, seenIds, nil
	}

	//TODO filter(cls.suitable_station, Station.code_manager.get_list_by_code(title))

	return result, seenIds, nil
}
