package server

import (
	"encoding/json"
	"fmt"
	"net/http"

	"github.com/opentracing/opentracing-go"

	"a.yandex-team.ru/travel/rasp/wizards/go/wizard_proxy_api/internal/direction"
	"a.yandex-team.ru/travel/rasp/wizards/go/wizard_proxy_api/internal/geomodel"
	"a.yandex-team.ru/travel/rasp/wizards/go/wizard_proxy_api/internal/utils"
)

type GeosearchResponse struct {
	DeparturePoint geomodel.PointRepr `json:"departure_point"`
	ArrivalPoint   geomodel.PointRepr `json:"arrival_point"`
}

func (server *Server) Geosearch(w http.ResponseWriter, r *http.Request) {
	geosearchSpan, _ := opentracing.StartSpanFromContext(r.Context(), "Geosearch")
	defer geosearchSpan.Finish()

	server.app.Logger.Info(fmt.Sprintf("request URL: %v", r.URL))
	query := r.URL.Query()

	queryParams := &direction.FindPointsParameters{
		Cfg:                server.app.Cfg,
		Storage:            server.app.Storage,
		TransportCode:      query.Get("transport"),
		DeparturePointName: query.Get("point"),
		ArrivalPointName:   query.Get("point_to"),

		ClientGeoid:              utils.ReadIntParameter(query.Get("geo_id")),
		DepartureSettlementGeoid: utils.ReadIntParameter(query.Get("departure_settlement_geoid")),
		ArrivalSettlementGeoid:   utils.ReadIntParameter(query.Get("arrival_settlement_geoid")),
	}

	departurePoint, arrivalPoint, err := direction.FindPoints(queryParams)
	if err != nil {
		w.WriteHeader(200)
		_, err := w.Write([]byte(fmt.Sprintf("{error: %v}", err)))
		if err != nil {
			server.app.Logger.Error(fmt.Sprintf("Cannot write answer: %v", err))
		}
	}

	dump, err := json.Marshal(GeosearchResponse{
		DeparturePoint: departurePoint.ReprPoint(),
		ArrivalPoint:   arrivalPoint.ReprPoint(),
	})
	if err != nil {
		w.WriteHeader(200)
		_, err := w.Write([]byte(fmt.Sprintf("{error: %v}", err)))
		if err != nil {
			server.app.Logger.Error(fmt.Sprintf("Cannot write answer: %v", err))
		}
	}
	w.WriteHeader(200)
	_, err = w.Write(dump)
	if err != nil {
		server.app.Logger.Error(fmt.Sprintf("Cannot write answer: %v", err))
	}
}
