package server

import (
	"context"
	"fmt"
	"io"
	"net/http"
	"time"

	"github.com/go-chi/chi/v5"
	chiMiddleware "github.com/go-chi/chi/v5/middleware"

	"a.yandex-team.ru/library/go/httputil/middleware/httpmetrics"
	"a.yandex-team.ru/travel/library/go/metrics"
	"a.yandex-team.ru/travel/library/go/tracing"
	"a.yandex-team.ru/travel/rasp/wizards/go/wizard_proxy_api/internal/application"
	"a.yandex-team.ru/travel/rasp/wizards/go/wizard_proxy_api/internal/middleware"
)

type Config struct {
	Port                       string        `config:"port,required"`
	PerfMetricsRefreshInterval time.Duration `config:"PERF_METRICS_REFRESH_INTERVAL,required"`
	PerfMetricsPrefix          string
}

type Server struct {
	cfg          *Config
	app          *application.Application
	router       *chi.Mux
	tracerCloser io.Closer
}

func (server *Server) Ping(w http.ResponseWriter, r *http.Request) {
	server.app.Logger.Info(fmt.Sprintf("request URL: %v", r.URL))
	w.WriteHeader(200)
	_, err := w.Write([]byte("Ok"))
	if err != nil {
		server.app.Logger.Error(fmt.Sprintf("Cannot write answer: %v", err))
	}
}

func New(ctx context.Context, cfg *Config, app *application.Application) (*Server, error) {
	server := &Server{
		cfg:          cfg,
		app:          app,
		router:       chi.NewRouter(),
		tracerCloser: tracing.InitializeDefaultTracer("wizard_proxy_api"),
	}

	tracingMiddleware := tracing.NewTracingMiddlewareBuilder().
		WithExtractor(tracing.NewHeaderTagExtractor("X-Request-Id", "request-id")).
		Build()

	httpMetricsRegistry := metrics.NewAppMetricsRegistryWithPrefix("http")
	httpMetrics := metrics.NewAppMetrics(httpMetricsRegistry)
	metrics.SetGlobalAppMetrics(httpMetrics)

	apiMiddleware := middleware.New(app.Logger)

	server.router.Use(apiMiddleware.Logger)
	server.router.Use(chiMiddleware.RequestID)
	server.router.Use(tracingMiddleware.Handle)
	server.router.Use(apiMiddleware.Recover)
	server.router.Use(httpmetrics.New(httpMetricsRegistry, httpmetrics.WithPathEndpoint()))

	server.router.Get("/geosearch", server.Geosearch)
	server.router.Get("/ping", server.Ping)
	server.router.Get("/proxy", server.Proxy)

	perfMetricsHandler := metrics.NewPerfMetricsHandler(server.cfg.PerfMetricsPrefix, server.cfg.PerfMetricsRefreshInterval)
	httpMetricsHandler := metrics.NewMetricsHandler("http", httpMetricsRegistry)
	server.router.Get(httpMetricsHandler.GetRoute(), httpMetricsHandler.Handle)
	server.router.Get(perfMetricsHandler.GetRoute(), perfMetricsHandler.Handle)

	return server, nil
}

func (server *Server) ListenAndServe() error {
	port := server.cfg.Port
	server.app.Logger.Info(fmt.Sprintf("Start listening at %v\n", port))

	err := http.ListenAndServe(":"+port, server.router)
	if err != nil {
		server.app.Logger.Error(fmt.Sprintf("Server stopped listening with error: %v\n", err))
	}
	return err
}

func (server *Server) Close() error {
	return server.tracerCloser.Close()
}
