# coding: utf-8
from __future__ import absolute_import, division, print_function, unicode_literals

from datetime import timedelta
from functools import partial

from django.utils import translation
from django.utils.http import urlencode

from travel.rasp.library.python.common23.date import environment
from common.utils.date import timedelta2minutes
from travel.library.python.tracing.instrumentation import traced_function
from travel.rasp.wizards.proxy_api.lib.direction.serialization import dump_direction_segments
from travel.rasp.wizards.proxy_api.lib.tracking import get_tracking_query_item, TrackingParameter
from travel.rasp.wizards.wizard_lib.experiment_flags import ExperimentFlag
from travel.rasp.wizards.wizard_lib.serialization.price import dump_price
from travel.rasp.wizards.wizard_lib.serialization.thread_express_type import ThreadExpressType
from travel.rasp.wizards.wizard_lib.utils.text import render_texts
from travel.rasp.wizards.wizard_lib.views.helpers.service_urls import format_morda_url, format_touch_url, get_morda_host


def get_search_urls(found_departure_date, query, url_query, transport_code=None):
    search_url_path, search_url_query = (
        (
            b'/search/{}/next/'.format(transport_code) if transport_code is not None else b'/search/next/',
            url_query
        )
        if query.departure_date is None and ExperimentFlag.DIRECTION_SEARCH_NEXT_URL in query.experiment_flags else
        (
            b'/search/{}/'.format(transport_code) if transport_code is not None else b'/search/',
            b'{url_query}&when={when}'.format(url_query=url_query, when=found_departure_date)
        )
    )
    return {
        'url': format_morda_url(search_url_path, search_url_query, query.tld),
        'touch_url': format_touch_url(search_url_path, search_url_query, query.tld),
    }


def _get_content(direction_data, transport_code):
    return {
        'departure_date': direction_data.found_departure_date.strftime('%Y-%m-%d'),
        'minimum_duration': timedelta2minutes(direction_data.minimum_duration),
        'minimum_price': dump_price(direction_data.minimum_price),
        'segments': dump_direction_segments(direction_data.segments),
        'title': translation.ugettext('by_{}'.format(transport_code)).capitalize(),
        'total': direction_data.total,
        'transport': transport_code,
    }


def _get_texts(query, transports_data):
    minimum_duration, minimum_duration_transport_code = min(
        (direction_data.minimum_duration, transport_code)
        for transport_code, direction_data in transports_data.items()
    )
    context = {
        'departure_point': query.departure_point,
        'arrival_point': query.arrival_point,
        'departure_date': query.departure_date,
        'minimum_duration': minimum_duration,
        'minimum_duration_transport_code': minimum_duration_transport_code,
        'transport_codes': tuple(transports_data)
    }
    return render_texts('direction', context, query.experiment_flags)


def _get_transport_texts(direction_data, query, transport_code):
    is_the_same_arrival = len(set(s.arrival_station.id for s in direction_data.segments)) == 1
    is_the_same_departure = len(set(s.departure_station.id for s in direction_data.segments)) == 1
    first_segment = direction_data.segments[0]
    context = {
        'departure_point': query.departure_point,
        'arrival_point': query.arrival_point,
        'snippet_departure_point': first_segment.departure_station if is_the_same_departure else query.departure_point,
        'snippet_arrival_point': first_segment.arrival_station if is_the_same_arrival else query.arrival_point,
        'departure_date': query.departure_date,
        'minimum_duration': direction_data.minimum_duration,
        'minimum_price': direction_data.minimum_price,
        'first_segment': first_segment,
        'today': environment.today(),
    }

    thread_express_type = query.thread_express_type
    template_code = transport_code
    if thread_express_type:
        template_code += '_' + thread_express_type.value

    return render_texts('direction_{}'.format(template_code), context, query.experiment_flags)


def _get_tomorrow_date(local_tz):
    return (environment.now_aware() + timedelta(days=1)).astimezone(local_tz).date()


def _get_transport_links(local_tz, tld, transport_code, url_query):
    tomorrow_url_query = b'{query}&when={when}'.format(query=url_query, when=_get_tomorrow_date(local_tz))
    search_url_path = b'/search/{}/'.format(transport_code)
    next_url_path = b'/search/{}/next/'.format(transport_code)
    return {
        'next_link': {
            'title': translation.ugettext('next_link_title'),
            'touch_url': format_touch_url(next_url_path, url_query, tld),
            'url': format_morda_url(next_url_path, url_query, tld),
        },
        'tomorrow_link': {
            'title': translation.ugettext('tomorrow_link_title'),
            'touch_url': format_touch_url(search_url_path, tomorrow_url_query, tld),
            'url': format_morda_url(search_url_path, tomorrow_url_query, tld),
        },
    }


def _get_deprecated_filter(thread_express_type, transport_code):
    if transport_code == 'suburban' and thread_express_type is not None:
        return thread_express_type.value
    return None


def get_thread_type_query_item(query):
    if (
            query.transport_code != 'suburban' or
            query.thread_express_type is None
    ):
        return []

    return [
        ('express', 'y')
        if query.thread_express_type == ThreadExpressType.EXPRESS
        else ('aeroex', 'y')
    ]


def _get_direction_url_query(query):
    parameters = [
            get_tracking_query_item(TrackingParameter.DIRECTION),
            ('fromName', query.departure_point.L_title()),
            ('fromId', query.departure_point.point_key),
            ('toName', query.arrival_point.L_title()),
            ('toId', query.arrival_point.point_key),
        ]
    parameters.extend(get_thread_type_query_item(query))
    if query.main_reqid is not None:
        parameters.append(('wizardReqId', query.main_reqid))
    return urlencode(parameters)


@traced_function
def format_direction(transports_data, query, default_transport_code=None):
    translation.activate(query.language)

    search_urls_factory = partial(
        get_search_urls,
        query=query,
        url_query=_get_direction_url_query(query),
    )

    search_urls = search_urls_factory(
        found_departure_date=min(
            direction_data.found_departure_date
            for direction_data in transports_data.values()
        ),
    )

    response = {
        'content': tuple(
            dict(
                _get_content(direction_data=direction_data, transport_code=transport_code),
                **search_urls_factory(
                    found_departure_date=direction_data.found_departure_date,
                    transport_code=transport_code
                )
            )
            for transport_code, direction_data in transports_data.items()
            if transport_code != default_transport_code
        ),
        'path_items': (dict(search_urls, **{'text': get_morda_host(query.tld)}),),
        'type': 'transports_with_default' if default_transport_code is not None else 'transports',
    }
    response.update(search_urls)

    if default_transport_code is not None:
        response.update({
            'default_transport': dict(
                _get_content(direction_data=transports_data[default_transport_code],
                             transport_code=default_transport_code),
                **search_urls_factory(
                    found_departure_date=transports_data[default_transport_code].found_departure_date,
                    transport_code=default_transport_code,
                )
            )
        })

    response.update(_get_texts(
        query=query,
        transports_data=transports_data,
    ))

    return response


@traced_function
def format_transport_direction(direction_data, query, transport_code):
    translation.activate(query.language)

    direction_url_query = _get_direction_url_query(query)
    search_urls = get_search_urls(
        found_departure_date=direction_data.found_departure_date,
        query=query,
        transport_code=transport_code,
        url_query=direction_url_query,
    )

    response = {
        'default_transport': dict(
            _get_content(direction_data=direction_data, transport_code=transport_code),
            **search_urls
        ),
        'path_items': (dict(search_urls, **{'text': get_morda_host(query.tld)}),),
        'type': 'transports_with_default',
        'filter': None,
        'content': None,
    }
    filter_value = _get_deprecated_filter(query.thread_express_type, transport_code)
    if filter_value is not None:
        response['filter'] = {filter_value: True}

    response.update(search_urls)

    response.update(_get_transport_texts(
        direction_data=direction_data,
        query=query,
        transport_code=transport_code,
    ))
    response.update(_get_transport_links(
        local_tz=query.departure_point.pytz,
        tld=query.tld,
        transport_code=transport_code,
        url_query=direction_url_query,
    ))

    return response
