# coding: utf-8
from __future__ import absolute_import, division, print_function, unicode_literals

from copy import copy

from common.models.geo import Settlement
from geosearch.views.point import PointSearchError, PointSearch, SuburbanPointSearch
from geosearch.views.pointlist import PointList
from geosearch.views.pointtopoint import process_points_lists as geosearch_process_points_lists, SamePointError
from travel.library.python.tracing.instrumentation import traced_function


@traced_function
def process_points_lists(*args, **kwargs):
    return geosearch_process_points_lists(*args, **kwargs)


def _get_settlement_by_geo_id(geo_id):
    try:
        return Settlement.hidden_manager.get(_geo_id=geo_id) if geo_id else None
    except (Settlement.DoesNotExist, ValueError):
        return None


class PointsException(Exception):
    pass


class PointNotFound(PointsException):
    pass


class PointsAreInvalid(PointsException):
    pass


@traced_function
def find_points(
    departure_settlement_geoid,
    departure_point_name,
    arrival_settlement_geoid,
    arrival_point_name,
    client_geoid=None,
    transport_code=None
):
    is_suburban_search = (transport_code == 'suburban')
    search = SuburbanPointSearch if is_suburban_search else PointSearch

    try:
        departure_point = _get_settlement_by_geo_id(departure_settlement_geoid)
        departure_pointlist = (
            PointList(departure_point, exact_variant=True) if departure_point else
            search.find_point(departure_point_name, t_type=transport_code)
        )

        arrival_point = _get_settlement_by_geo_id(arrival_settlement_geoid)
        arrival_pointlist = (
            PointList(arrival_point, exact_variant=True) if arrival_point else
            search.find_point(arrival_point_name, t_type=transport_code)
        )
    except PointSearchError as error:
        raise PointNotFound(error)

    try:
        client_city = _get_settlement_by_geo_id(client_geoid)

        process_points_lists(
            copy(departure_pointlist),
            copy(arrival_pointlist),
            client_city=client_city,
            suburban=is_suburban_search
        )

        departure_pointlist, arrival_pointlist = process_points_lists(
            departure_pointlist,
            arrival_pointlist,
            client_city=client_city,
            disable_replace=True,
            suburban=is_suburban_search
        )
    except SamePointError as error:
        raise PointsAreInvalid(error)

    return departure_pointlist.point, arrival_pointlist.point
