# coding: utf-8
from __future__ import absolute_import, division, print_function, unicode_literals

import itertools

from rest_framework import serializers

from common.models.currency import Currency, Price
from travel.rasp.wizards.proxy_api.lib.direction.models import DirectionData, DirectionQuery, Segment
from travel.rasp.wizards.proxy_api.lib.segments.serialization import dump_segments, load_segments
from travel.rasp.wizards.proxy_api.lib.tracking import TrackingParameter, add_tracking_parameter
from travel.rasp.wizards.wizard_lib.protobuf_models.direction_response_pb2 import TDirectionResponse
from travel.rasp.wizards.wizard_lib.serialization.date import parse_date
from travel.rasp.wizards.wizard_lib.serialization.language import parse_language
from travel.rasp.wizards.wizard_lib.serialization.limit import parse_limit
from travel.rasp.wizards.wizard_lib.serialization.price import dump_price
from travel.rasp.wizards.wizard_lib.serialization.thread_express_type import parse_thread_express_type


def load_direction_query(departure_point, arrival_point, transport_code, experiment_flags, query_params):
    try:
        thread_express_type = parse_thread_express_type(query_params.get('query'))
    except serializers.ValidationError:
        thread_express_type = None

    main_reqid = query_params.get('main_reqid') or None

    return DirectionQuery(
        departure_point=departure_point,
        arrival_point=arrival_point,
        transport_code=transport_code,
        departure_date=parse_date(query_params.get('date'), departure_point.pytz),
        thread_express_type=thread_express_type,
        thread_number=query_params.get('number'),
        language=parse_language(query_params.get('lang')),
        experiment_flags=experiment_flags,
        tld=query_params.get('tld'),
        limit=parse_limit(query_params.get('limit'), experiment_flags),
        main_reqid=main_reqid,
    )


def _load_price(proto_price):
    currency = Currency.objects.get(id=proto_price.CurrencyId)
    return Price(proto_price.Value, currency.code)


def load_direction_data(raw_direction_data):
    proto_direction = TDirectionResponse()
    proto_direction.ParseFromString(raw_direction_data)
    return DirectionData(
        segments=load_segments(
            proto_direction.Segments,
            segment_factory=Segment,
            extra_kwargs_getter=lambda proto_segment: dict(
                price=_load_price(proto_segment.Price) if proto_segment.HasField('Price') else None
            )
        ),
        total=proto_direction.Total
    )


def dump_direction_segments(segments):
    result = dump_segments(segments)
    for segment, segment_dump in itertools.izip(segments, result):
        urls = segment.urls
        segment_dump.update({
            'price': dump_price(segment.price),
            'touch_url': add_tracking_parameter(urls.mobile, TrackingParameter.DIRECTION),
            'url': add_tracking_parameter(urls.desktop, TrackingParameter.DIRECTION),
        })
    return result
