# coding: utf-8
from __future__ import absolute_import, division, print_function, unicode_literals

from django.utils import translation

from common.models.transport import TransportType
from travel.library.python.tracing.instrumentation import traced_function
from travel.rasp.wizards.proxy_api.lib.tracking import get_tracking_query_parameter, TrackingParameter
from travel.rasp.wizards.wizard_lib.experiment_flags import ExperimentFlag
from travel.rasp.wizards.wizard_lib.serialization.intent_type import IntentType
from travel.rasp.wizards.wizard_lib.utils.text import wrap_for_highlight
from travel.rasp.wizards.wizard_lib.views.helpers.service_urls import (
    format_morda_url, format_touch_url, format_trains_url, get_morda_host, get_trains_host
)


def _is_trains_query(query):
    return (
        query.transport_code == 'train'
        and
        (
            ExperimentFlag.TRAIN_LINK_ALL in query.experiment_flags
            or
            ExperimentFlag.TRAIN_LINK_BUY_INTENT in query.experiment_flags and query.intent == IntentType.BUY
        )
    )


def _format_morda_and_touch_urls(path, tracking_query, tld):
    return {
        'url': format_morda_url(path, tracking_query, tld),
        'touch_url': format_touch_url(path, tracking_query, tld)
    }


def _format_urls(path, tracking_query, query):
    if _is_trains_query(query):
        train_url = format_trains_url(path, tracking_query, query.tld)
        return {
            'url': train_url,
            'touch_url': train_url
        }

    return _format_morda_and_touch_urls(path, tracking_query, query.tld)


def _get_host_for_path_items(query):
    if _is_trains_query(query):
        return get_trains_host(query.tld)

    return get_morda_host(query.tld)


def _get_transports_content(departure_settlement, query_transport_code, tld):
    departure_settlement_type_choices = departure_settlement.type_choices_set
    transport_codes = [
        transport_code
        for transport_code in TransportType.list_national_codes(tld)
        if transport_code in departure_settlement_type_choices
    ]
    selected_transport_code = query_transport_code if query_transport_code in transport_codes else None

    return [
        {
            'code': transport_code,
            'title': TransportType.objects.get(code=transport_code).L_title(),
            'selected': transport_code == selected_transport_code
        }
        for transport_code in transport_codes
    ]


def get_general_urls(departure_settlement, tld, tracking_query):
    path = b'/city/{}'.format(departure_settlement.id)
    return _format_morda_and_touch_urls(path, tracking_query, tld)


@traced_function
def format_general_response(query):
    translation.activate(query.language)

    departure_settlement, departure_date = query.departure_settlement, query.departure_date

    content = {
        'departure_settlement': {
            'key': departure_settlement.point_key,
            'title': departure_settlement.L_title()
        },
        'transports': _get_transports_content(departure_settlement, query.transport_code, query.tld)
    }

    if departure_date is not None:
        content['date'] = unicode(departure_date)

    tracking_query = get_tracking_query_parameter(TrackingParameter.GENERAL)

    index_urls = _format_urls('/', tracking_query, query)
    settlement_urls = get_general_urls(departure_settlement, query.tld, tracking_query)
    path_items = [dict(index_urls, **{'text': _get_host_for_path_items(query)})]

    not_is_trains_query = not _is_trains_query(query)
    if not_is_trains_query:
        path_items.append(dict(settlement_urls, **{'text': departure_settlement.L_title()}))

    response = {
        'type': 'direction_query',
        'path_items': path_items,
        'title': wrap_for_highlight(
            translation.ugettext('schedule_of_settlement') % {
                'settlement_genitive_title': departure_settlement.L_title(case='genitive')
            }
        ),
        'content': content
    }
    if not_is_trains_query:
        response.update(settlement_urls)
    else:
        response.update(index_urls)

    return response
