# coding: utf-8
from __future__ import absolute_import, division, print_function, unicode_literals

from django.conf import settings

from common.models.geo import Settlement, Station
from common.models.transport import TransportType
from common.models_abstract.schedule import ExpressType
from common.settings.utils import define_setting, bool_converter
from travel.library.python.tracing.instrumentation import traced_function
from travel.rasp.wizards.proxy_api.lib.direction.points import find_points, PointsException
from travel.rasp.wizards.proxy_api.lib.direction.serialization import load_direction_query
from travel.rasp.wizards.proxy_api.lib.general.serialization import load_general_query
from travel.rasp.wizards.proxy_api.lib.logger import log_run_time, logger
from travel.rasp.wizards.proxy_api.lib.station.search import find_aeroexpress_station, find_station
from travel.rasp.wizards.proxy_api.lib.station.serialization import load_plane_station_query, load_suburban_station_query
from travel.rasp.wizards.wizard_lib.experiment_flags import ExperimentFlag


for key in (
    'DISABLE_DIRECTION_WIZARD',
    'DISABLE_STATION_WIZARD',
    'DISABLE_ROUTE_WIZARD',
    'DISABLE_GENERAL_WIZARD',
):
    define_setting(key, default=False, converter=bool_converter)


class NoContent(Exception):
    pass


def _get_regular_direction_query(
    departure_settlement_geoid,
    departure_point_name,
    arrival_settlement_geoid,
    arrival_point_name,
    transport_code,
    experiment_flags,
    query_params
):
    client_geoid = query_params.get('geo_id')

    if transport_code == 'train':  # 'train' is synonym of 'suburban' in legacy wizard
        transport_code = 'suburban'

    try:
        with log_run_time('_get_regular_direction_query find_points'):
            departure_point, arrival_point = find_points(
                departure_settlement_geoid=departure_settlement_geoid,
                departure_point_name=departure_point_name,
                arrival_settlement_geoid=arrival_settlement_geoid,
                arrival_point_name=arrival_point_name,
                client_geoid=client_geoid,
                transport_code=transport_code
            )
    except PointsException:
        raise NoContent

    if not (
        isinstance(departure_point, (Settlement, Station,)) and isinstance(arrival_point, (Settlement, Station,))
    ):
        raise NoContent

    return load_direction_query(departure_point, arrival_point, transport_code, experiment_flags, query_params)


@traced_function
def _get_aeroexpress_direction_query(
    experiment_flags, query_params, departure_station_name=None, arrival_station_name=None
):
    if not (bool(departure_station_name) ^ bool(arrival_station_name)):
        raise ValueError('one of arguments have to be specified: departure_station_name or arrival_station_name')

    with log_run_time('_get_aeroexpress_direction_query find_aeroexpress_station'):
        station = find_aeroexpress_station(departure_station_name or arrival_station_name)

    if station is None or not station.settlement_id:
        raise NoContent

    departure_point, arrival_point = (
        (station, station.settlement) if departure_station_name else (station.settlement, station)
    )
    return load_direction_query(departure_point, arrival_point, 'suburban', experiment_flags, query_params)


@traced_function
def get_direction_query(query_params, experiment_flags):
    if settings.DISABLE_DIRECTION_WIZARD or ExperimentFlag.ENABLE_DIRECTION_WIZARD not in experiment_flags:
        return

    departure_point_name = query_params.get('point')
    departure_settlement_geoid = query_params.get('departure_settlement_geoid')
    arrival_point_name = query_params.get('point_to')
    arrival_settlement_geoid = query_params.get('arrival_settlement_geoid')
    transport_code = query_params.get('transport') or None

    if (departure_point_name or departure_settlement_geoid) and (arrival_point_name or arrival_settlement_geoid):
        with log_run_time('_get_regular_direction_query'):
            return _get_regular_direction_query(
                departure_settlement_geoid=departure_settlement_geoid,
                departure_point_name=departure_point_name,
                arrival_settlement_geoid=arrival_settlement_geoid,
                arrival_point_name=arrival_point_name,
                transport_code=transport_code,
                experiment_flags=experiment_flags,
                query_params=query_params
            )

    logger.debug('start _get_aeroexpress_direction_query')
    if transport_code == 'suburban' and query_params.get('query') == ExpressType.AEROEXPRESS:
        if ExperimentFlag.BIDIRECTIONAL_AEROEXPRESS in experiment_flags:
            if departure_point_name:
                with log_run_time('_get_aeroexpress_direction_query, case - departure_point_name'):
                    return _get_aeroexpress_direction_query(
                        departure_station_name=departure_point_name,
                        experiment_flags=experiment_flags,
                        query_params=query_params
                    )
            elif arrival_point_name:
                with log_run_time('_get_aeroexpress_direction_query, case - arrival_point_name'):
                    return _get_aeroexpress_direction_query(
                        arrival_station_name=arrival_point_name,
                        experiment_flags=experiment_flags,
                        query_params=query_params
                    )
        elif departure_point_name:
            with log_run_time('_get_aeroexpress_direction_query, case - not ExperimentFlag.BIDIRECTIONAL_AEROEXPRESS'):
                return _get_aeroexpress_direction_query(
                    arrival_station_name=departure_point_name,
                    experiment_flags=experiment_flags,
                    query_params=query_params
                )


@traced_function
def get_station_query(query_params, experiment_flags):
    if settings.DISABLE_STATION_WIZARD or ExperimentFlag.ENABLE_STATION_WIZARD not in experiment_flags:
        return

    station_name = query_params.get('point')
    if station_name:
        with log_run_time('get_station_query find_station'):
            station = find_station(
                station_name,
                transport_code=query_params.get('transport') or 'suburban',
                client_geoid=query_params.get('geo_id') or None
            )

        if station is None:
            raise NoContent

        if station.t_type_id == TransportType.PLANE_ID:
            return load_plane_station_query(station, experiment_flags, query_params)

        return load_suburban_station_query(station, experiment_flags, query_params)


@traced_function
def get_route_query(query_params, experiment_flags):
    if settings.DISABLE_ROUTE_WIZARD or ExperimentFlag.ENABLE_ROUTE_WIZARD not in experiment_flags:
        return

    if query_params.get('brand') or query_params.get('number'):
        # RASPWIZARDS-593: Закрыть колдуны: поиск по рейсу и поиск по бренду
        raise NoContent


@traced_function
def get_general_query(query_params, experiment_flags):
    if settings.DISABLE_GENERAL_WIZARD or ExperimentFlag.ENABLE_GENERAL_WIZARD not in experiment_flags:
        return

    geoid = query_params.get('geo_id')
    if geoid:
        try:
            with log_run_time('get_general_query Settlement.hidden_manager.get()'):
                departure_settlement = Settlement.hidden_manager.get(_geo_id=geoid)
        except Settlement.DoesNotExist:
            raise NoContent

        return load_general_query(departure_settlement, experiment_flags, query_params)
