# coding: utf-8
from __future__ import absolute_import, division, print_function, unicode_literals

import itertools
from threading import local

import requests
from concurrent.futures import ThreadPoolExecutor
from django.conf import settings
from opentracing import global_tracer

from common.data_api.baris.instance import create_baris_instance


class _RequestsPoolLocals(local):
    def __init__(self):
        self.session = requests.Session()


class RequestsPool(object):
    empty = True

    def __init__(self, size):
        self._size = size
        self._executor = None
        self._baris_client = None
        self._locals = _RequestsPoolLocals()
        self.cleanup()

    def _get_session(self):
        return self._locals.session

    def _make_request(self, url, raise_for_status):
        response = self._get_session().get(url, timeout=settings.PROXY_API_REQUEST_SOURCES_TIMEOUT)
        if raise_for_status:
            response.raise_for_status()
        return response

    def _fetch_url(self, url, raise_for_status, active_span):
        if active_span is not None:
            with global_tracer().scope_manager.activate(active_span, True) as scope:
                return self._make_request(url, raise_for_status)
        return self._make_request(url, raise_for_status)

    def fetch_urls(self, urls, raise_for_status=True):
        self.empty = False
        active_span = getattr(global_tracer().scope_manager.active, 'span', None)
        return self._executor.map(
            self._fetch_url,
            urls,
            itertools.repeat(raise_for_status),
            itertools.repeat(active_span))

    def get_baris_station_tablo(
        self, station_id, direction='departure', after=None, before=None, limit=None, terminal=None
    ):
        self.empty = False
        return self._baris_client.get_station_tablo(
            station_id=station_id,
            direction=direction,
            after=after,
            before=before,
            limit=limit,
            terminal=terminal
        )

    def cleanup(self):
        # ThreadPoolExecutor can't be reused after shutdown, we have to create it again
        if not self.empty:
            self._executor.shutdown()
            del self.empty
        self._executor = ThreadPoolExecutor(max_workers=self._size)
        self._baris_client = create_baris_instance(
            host=settings.BARIS_API_URL,
            timeout=settings.PROXY_API_REQUEST_SOURCES_TIMEOUT,
            retry_config=None
        )


default_pool = RequestsPool(size=settings.PROXY_API_REQUESTS_POOL_SIZE)
fetch_urls = default_pool.fetch_urls
get_baris_station_tablo = default_pool.get_baris_station_tablo
