# coding: utf-8
from __future__ import absolute_import, division, print_function, unicode_literals

from datetime import datetime, time, timedelta
from collections import OrderedDict

from django.conf import settings
from requests.exceptions import ReadTimeout
from rest_framework import status
from rest_framework.response import Response
from singledispatch import singledispatch
from six import text_type

from common.data_api.baris.helpers import BarisData
from travel.rasp.library.python.common23.date.environment import now_aware
from travel.library.python.tracing.instrumentation import child_span, traced_function
from travel.rasp.wizards.proxy_api.lib.api_urls import (
    format_bus_direction_url, format_suburban_direction_url, format_suburban_station_url
)
from travel.rasp.wizards.proxy_api.lib.direction.aggregation import aggregate_direction_data
from travel.rasp.wizards.proxy_api.lib.direction.formats import format_direction, format_transport_direction
from travel.rasp.wizards.proxy_api.lib.direction.models import DirectionQuery
from travel.rasp.wizards.proxy_api.lib.direction.serialization import load_direction_data
from travel.rasp.wizards.proxy_api.lib.general.formats import format_general_response
from travel.rasp.wizards.proxy_api.lib.general.models import GeneralQuery
from travel.rasp.wizards.proxy_api.lib.logger import log_run_time, logger
from travel.rasp.wizards.proxy_api.lib.query_parsing import (
    get_direction_query, get_general_query, get_route_query, get_station_query, NoContent
)
from travel.rasp.wizards.proxy_api.lib.requests_pool import get_baris_station_tablo, fetch_urls
from travel.rasp.wizards.proxy_api.lib.station.formats import format_plane_station, format_suburban_station
from travel.rasp.wizards.proxy_api.lib.station.models import PlaneStationQuery, SuburbanStationQuery
from travel.rasp.wizards.proxy_api.lib.station.serialization import (
    load_suburban_station_data, PLANE_DIRECTION_TYPE_TO_VALUE
)
from travel.rasp.wizards.wizard_lib.serialization.experiment_flags import parse_experiment_flags
from travel.rasp.wizards.wizard_lib.station.direction_type import DirectionType


DIRECTION_URL_FORMATTERS = OrderedDict((
    ('suburban', format_suburban_direction_url),
    ('bus', format_bus_direction_url),
))


@singledispatch
def get_query_response(query, query_params):
    raise TypeError('Unknown query type: {!r}'.format(query))


def _get_direction_source_urls(query, query_params):
    source_urls = tuple(
        (source_transport_code, url_formatter(query, query_params))
        for source_transport_code, url_formatter in DIRECTION_URL_FORMATTERS.items()
        if not (query.thread_express_type and source_transport_code != 'suburban')
    )

    default_transport_code = query.transport_code
    if query.transport_code is not None and \
            not any(transport_code == default_transport_code for transport_code, _ in source_urls):
        return

    return source_urls


def _make_direction_response_for_error(source_url, transport_code, error, status_code):
    return Response(
        {
            'error': 'cannot get direction response',
            'exception': text_type(error),
            'transport_code': transport_code,
            'url': source_url
        },
        status=status_code,
    )


@traced_function
def make_direction_response(query, source_urls):
    with log_run_time('make_direction_response fetch_urls'):
        source_responses = fetch_urls(source_url for _transport_code, source_url in source_urls)
    default_transport_code = query.transport_code
    transports_data = OrderedDict()
    with child_span('proxy_api.lib.router.make_direction_response::fetch_and_process_source_urls'):
        for transport_code, source_url in source_urls:
            try:
                source_response = next(source_responses)
            except ReadTimeout as error:
                return _make_direction_response_for_error(source_url, transport_code, error, status.HTTP_200_OK)
            except Exception as error:
                return _make_direction_response_for_error(source_url, transport_code, error, status.HTTP_502_BAD_GATEWAY)

            if source_response.status_code != status.HTTP_204_NO_CONTENT:
                direction_data = load_direction_data(source_response.content)
                transports_data[transport_code] = aggregate_direction_data(direction_data)

    if not transports_data or \
            default_transport_code is not None and default_transport_code not in transports_data:
        return

    if len(transports_data) == 1:
        transport_code, direction_data = transports_data.items()[0]
        return Response(format_transport_direction(
            direction_data=direction_data,
            query=query,
            transport_code=transport_code,
        ))

    return Response(format_direction(
        transports_data=transports_data,
        query=query,
        default_transport_code=default_transport_code
    ))


@get_query_response.register(DirectionQuery)
@traced_function
def get_direction_response(query, query_params):
    with log_run_time('get_direction_response _get_direction_source_urls'):
        source_urls = _get_direction_source_urls(query, query_params)
    if source_urls:
        with log_run_time('get_direction_response make_direction_response'):
            return make_direction_response(query, source_urls)


def _make_suburban_response_for_error(url, error, status_code):
    return Response(
        {
            'error': 'cannot get suburban station response',
            'exception': text_type(error),
            'url': url
        },
        status=status_code,
    )


@get_query_response.register(SuburbanStationQuery)
@traced_function
def get_suburban_station_response(query, query_params):
    url = format_suburban_station_url(query, query_params)
    try:
        with log_run_time('get_suburban_station_response fetch_urls'):
            source_response = next(fetch_urls([url]))  # fetching through a pool to reuse existing connections
    except ReadTimeout as error:
        return _make_suburban_response_for_error(url, error, status.HTTP_200_OK)
    except Exception as error:
        return _make_suburban_response_for_error(url, error, status.HTTP_502_BAD_GATEWAY)

    if source_response.status_code != status.HTTP_204_NO_CONTENT:
        with log_run_time('get_suburban_station_response load and format suburban station data'):
            data = load_suburban_station_data(source_response.content)
            return Response(format_suburban_station(data, query))


def _make_plane_response_for_error(url, error, status_code):
    return Response(
        {
            'error': 'cannot get plane station response',
            'exception': text_type(error),
            'url': url,
            'search_props': {
                'plane-connection': 1,
                'connection-timeout': 1,
            }
        },
        status=status_code,
    )


@get_query_response.register(PlaneStationQuery)
@traced_function
def get_plane_station_response(query, query_params):
    # Default behaviour: wizard shows only one tab anyway
    if not query.direction_type:
        query = query._replace(direction_type=DirectionType.ARRIVAL)

    after = (datetime.combine(query.event_date, time(0, 0, 0)) if query.event_date
             else now_aware().astimezone(query.station.pytz))
    before = after + timedelta(days=1)

    try:
        with log_run_time('get_plane_station_response get_baris_station_tablo'):
            baris_response = get_baris_station_tablo(
                query.station.id,
                direction=PLANE_DIRECTION_TYPE_TO_VALUE[query.direction_type],
                after=after,
                before=before,
                limit=settings.PLANE_STATION_MAXIMUM_SEGMENTS
            )
    except ReadTimeout as error:
        return _make_plane_response_for_error("<BARIS>/flight_board", error, status.HTTP_200_OK)
    except Exception as error:
        logger.debug('get_baris_station_tablo (station_id = {}) failed: {}'.format(query.station.id, error))
        return _make_plane_response_for_error("<BARIS>/flight_board", error, status.HTTP_502_BAD_GATEWAY)

    if baris_response.flights:  # exists and not empty
        with log_run_time('get_plane_station_response load and format plane station data'):
            baris_data = BarisData(baris_response)
            return Response(format_plane_station(baris_data, query))


@get_query_response.register(GeneralQuery)
@traced_function
def get_general_response(query, _query_params):
    with log_run_time('get_general_response'):
        return Response(format_general_response(query))


@traced_function
def get_response(query_params):
    experiment_flags = parse_experiment_flags(query_params.get('exp_flags'))

    for query_parser in (get_direction_query, get_station_query, get_route_query, get_general_query):
        try:
            with log_run_time(query_parser.__name__):
                query = query_parser(query_params, experiment_flags)
        except NoContent:
            return

        if query is not None:
            with log_run_time('get_query_response'):
                return get_query_response(query, query_params)
