# coding: utf-8
from __future__ import absolute_import, division, print_function, unicode_literals

from datetime import date, datetime

from dateutil.tz import tzoffset
from django.utils.translation import ugettext

from common.models.geo import Station
from common.models.schedule import Company
from common.models.transport import TransportSubtype
from common.utils.date import timedelta2minutes
from travel.rasp.wizards.proxy_api.lib.segments.models import BaseSegment, Thread, Urls
from travel.rasp.wizards.wizard_lib.protobuf_models.thread_pb2 import TThread
from travel.rasp.wizards.wizard_lib.serialization.thread_express_type import ThreadExpressType


DEFAULT_TRANSPORT_SUBTYPES = {
    TransportSubtype.TRAIN_ID, TransportSubtype.PLANE_ID, TransportSubtype.BUS_ID, TransportSubtype.SUBURBAN_ID
}


def _load_datettime(proto_local_timestamp):
    return datetime.fromtimestamp(proto_local_timestamp.Timestamp, tzoffset('', proto_local_timestamp.UtcOffset))


def _iparse_segments(proto_segments, segment_factory, extra_kwargs_getter):
    stations = Station.objects.in_bulk(
        station_id
        for proto_segment in proto_segments
        for station_id in (proto_segment.DepartureStationId, proto_segment.ArrivalStationId,)
    )
    transport_subtypes = {
        subtype.id: subtype for subtype in TransportSubtype.objects.all_cached()
    }
    companies = Company.objects.in_bulk(
        filter(None, (proto_segment.Thread.CompanyId for proto_segment in proto_segments))
    )

    for proto_segment in proto_segments:
        proto_thread = proto_segment.Thread
        yield segment_factory(
            thread=Thread(
                company=companies[proto_thread.CompanyId] if proto_thread.CompanyId else None,
                express_type=(
                    ThreadExpressType[TThread.EExpressType.Name(proto_thread.ExpressType)]
                    if proto_thread.ExpressType else
                    None
                ),
                number=proto_thread.Number,
                start_date=date.fromordinal(proto_thread.StartDate) if proto_thread.StartDate else None,
                title=proto_thread.Title,
                transport_subtype=(
                    transport_subtypes[proto_thread.TransportSubtypeId] if proto_thread.TransportSubtypeId else
                    None
                ),
            ),
            urls=Urls(proto_segment.Urls.Desktop, proto_segment.Urls.Mobile),
            departure_station=stations[proto_segment.DepartureStationId],
            departure_local_dt=_load_datettime(proto_segment.DepartureTimestamp),
            arrival_station=stations[proto_segment.ArrivalStationId],
            arrival_local_dt=_load_datettime(proto_segment.ArrivalTimestamp),
            **extra_kwargs_getter(proto_segment)
        )


def load_segments(proto_segments, segment_factory=BaseSegment, extra_kwargs_getter=lambda proto_segment: {}):
    return tuple(_iparse_segments(proto_segments, segment_factory, extra_kwargs_getter))


def _dump_datetime(local_dt):
    return local_dt.strftime('%Y-%m-%d %H:%M:%S %z')


def _dump_segment(segment):
    thread = segment.thread

    segment_dump = {
        'arrival_station': segment.arrival_station.L_popular_title(),
        'arrival': _dump_datetime(segment.arrival_local_dt),
        'departure': _dump_datetime(segment.departure_local_dt),
        'departure_station': segment.departure_station.L_popular_title(),
        'duration': timedelta2minutes(segment.arrival_local_dt - segment.departure_local_dt),
        'from_station': ugettext('from_station') % {
            'station_title': segment.departure_station.L_popular_title(case='genitive')
        },
        'to_station': ugettext('to_station') % {
            'station_title': segment.arrival_station.L_popular_title(case='genitive')
        },
        'number': thread.number,
        'title': thread.title,
        'transport_subtype_title': (
            thread.transport_subtype.L_title()
            if thread.transport_subtype and thread.transport_subtype.id not in DEFAULT_TRANSPORT_SUBTYPES else
            None
        ),
    }

    express_type = thread.express_type
    if express_type is not None:
        segment_dump[express_type.value] = True

    return segment_dump


def dump_segments(segments):
    return tuple(_dump_segment(segment) for segment in segments)
