# coding: utf-8
from __future__ import absolute_import, division, print_function, unicode_literals

from datetime import datetime

from dateutil import parser
from django.conf import settings
from django.utils import translation
from django.utils.http import urlencode
from django.utils.translation import ugettext, ugettext_lazy as _
from six import text_type
from urllib import quote

from common.models.geo import Direction, DirectionFromTranslate
from travel.library.python.tracing.instrumentation import traced_function
from travel.rasp.wizards.proxy_api.lib.station.models import EventUpdateStatus
from travel.rasp.wizards.proxy_api.lib.station.serialization import PLANE_DIRECTION_TYPE_TO_VALUE, dump_suburban_station_segments
from travel.rasp.wizards.proxy_api.lib.tracking import TrackingParameter, add_tracking_parameter, get_tracking_query_item
from travel.rasp.wizards.wizard_lib.station.direction_type import DirectionType
from travel.rasp.wizards.wizard_lib.utils.text import render_texts
from travel.rasp.wizards.wizard_lib.views.helpers.service_urls import format_morda_url, format_touch_url, get_morda_host
from travel.rasp.wizards.wizard_lib.experiment_flags import ExperimentFlag


DIRECTION_TITLE_TEXTS = {
    DirectionType.DEPARTURE: _('departure'),
    DirectionType.ARRIVAL: _('arrival'),
    DirectionType.ALL: _('all_directions'),
}
EVENT_UPDATE_TEXTS = {
    EventUpdateStatus.CANCELLED: _('cancelled'),
    EventUpdateStatus.ARRIVED: _('arrived'),
    EventUpdateStatus.DEPARTED: _('departed'),
    EventUpdateStatus.ON_TIME: _('expected'),
}
EVENT_UPDATE_EARLY_TEXTS = {
    DirectionType.DEPARTURE: _('early_departure'),
    DirectionType.ARRIVAL: _('early_arrival'),
}
EVENT_UPDATE_DELAYED_TEXTS = {
    DirectionType.DEPARTURE: _('delayed_departure'),
    DirectionType.ARRIVAL: _('late_arrival'),
}
BARIS_FLIGHT_STATUS_TO_TYPE = {
    'unknown': EventUpdateStatus.UNKNOWN,
    'on_time': EventUpdateStatus.ON_TIME,
    'delayed': EventUpdateStatus.DELAYED,
    'early': EventUpdateStatus.EARLY,
    'cancelled': EventUpdateStatus.CANCELLED,
    'departed': EventUpdateStatus.DEPARTED,
    'arrived': EventUpdateStatus.ARRIVED
}


def get_suburban_station_urls(direction_code, query):
    event_date = query.event_date
    station = query.station
    tld = query.tld

    return {
        'url': format_morda_url(
            b'/station/{}/'.format(station.id),
            urlencode((
                get_tracking_query_item(TrackingParameter.SUBURBAN_STATION),
                ('type', 'suburban'),
                ('span', 'day' if event_date is None else 'schedule'),
                ('direction', direction_code),
            )),
            tld
        ),
        'touch_url': format_touch_url(
            b'/station/{}/suburban/'.format(station.id),
            urlencode((
                get_tracking_query_item(TrackingParameter.SUBURBAN_STATION),
                ('filter',
                    'today' if (
                        event_date is None and
                        ExperimentFlag.SUBURBAN_STATION_TOUCH_TODAY_URL in query.experiment_flags
                    ) else 'all'),
                ('direction', direction_code),
            )),
            tld
        )
    }


def get_plane_station_urls(query, direction_type=None):
    station = query.station
    tld = query.tld

    station_path = b'/station/{}/'.format(station.id)
    station_params = [get_tracking_query_item(TrackingParameter.PLANE_STATION)]

    if not station.tablo_state:
        station_params.append(('span', 'day' if query.event_date is None else 'schedule'))

    if direction_type:
        station_params.append(('event', PLANE_DIRECTION_TYPE_TO_VALUE[direction_type]))

    return {
        'url': format_morda_url(station_path, urlencode(station_params), tld),
        'touch_url': format_touch_url(station_path, urlencode(station_params), tld)
    }


def _get_suburban_station_texts(data, query):
    context = {
        'event_date': query.event_date,
        'station': query.station
    }
    return render_texts('suburban_station', context, query.experiment_flags)


def _get_plane_station_texts(query):
    station = query.station
    direction_type = query.direction_type
    context = {
        'direction_type': PLANE_DIRECTION_TYPE_TO_VALUE[direction_type] if direction_type is not None else None,
        'event_date': query.event_date,
        'is_schedule': not station.tablo_state,
        'station': station
    }
    return render_texts('plane_station', context, query.experiment_flags)


def _get_direction_title(direction):
    return (
        Direction.objects.get(code=direction.code).L_title() if direction.type is DirectionType.DIR else
        DirectionFromTranslate.get(direction.code) if direction.type is DirectionType.SUBDIR else
        text_type(DIRECTION_TITLE_TEXTS[direction.type])
    )


def _get_tablo_status(station, direction_type, status=None):
    if station.tablo_state == 'nodata':
        return ugettext('no_data')

    if status is EventUpdateStatus.EARLY:
        return text_type(EVENT_UPDATE_EARLY_TEXTS[direction_type])

    if status is EventUpdateStatus.DELAYED:
        return text_type(EVENT_UPDATE_DELAYED_TEXTS[direction_type])

    status_text = EVENT_UPDATE_TEXTS.get(status)
    if status_text is not None:
        return text_type(status_text)

    return ugettext('no_data') if station.tablo_state else ugettext('expected')


def _get_tablo_flight_title(baris_data, baris_flight):
    first_station = baris_data.stations_by_ids[baris_flight['route'][0]]
    last_station = baris_data.stations_by_ids[baris_flight['route'][-1]]

    if first_station.settlement_id in baris_data.settlements_by_ids:
        first_point = baris_data.settlements_by_ids[first_station.settlement_id]
    else:
        first_point = first_station
    if last_station.settlement_id in baris_data.settlements_by_ids:
        last_point = baris_data.settlements_by_ids[last_station.settlement_id]
    else:
        last_point = last_station

    return '{} \N{em dash} {}'.format(first_point.L_title(), last_point.L_title())


def _event_update_status(baris_flight_status):
    if baris_flight_status in BARIS_FLIGHT_STATUS_TO_TYPE:
        return BARIS_FLIGHT_STATUS_TO_TYPE[baris_flight_status]
    else:
        return EventUpdateStatus.UNKNOWN


def _build_flight_avia_link(number, query):
    host = settings.NATIONAL_AVIA_DOMAINS.get(query.tld, settings.NATIONAL_AVIA_DOMAINS['ru'])
    number = quote(number.replace(' ', '-').encode('utf-8'))
    raw_url = 'https://{host}/flights/{number}/?lang={lang}'.format(
        host=host, number=number, lang=query.language
    )
    return raw_url


def _plane_station_title(station):
    if station is not None and station.settlement is not None:
        return station.settlement.L_title()
    return None


def _dump_baris_segment(flight, baris_data, query):
    url = _build_flight_avia_link(flight['title'], query)
    status = flight['status']

    event_update, status_update, terminal_update, gate = (
        (status['departure'], status['departureStatus'], status['departureTerminal'], status['departureGate'])
        if query.direction_type is DirectionType.DEPARTURE else
        (status['arrival'], status['arrivalStatus'], status['arrivalTerminal'], status['arrivalGate'])
    )
    scheduled_dt = parser.parse(flight['datetime'])
    expected_dt = (datetime.strptime(event_update, '%Y-%m-%d %H:%M:%S').replace(tzinfo=query.station.pytz)
                   if event_update
                   else scheduled_dt)

    flight_status = status_update if status_update else status['status']

    arrival_station = (
        baris_data.stations_by_ids[flight['stationTo']]
        if 'stationTo' in flight and flight['stationTo'] in baris_data.stations_by_ids
        else None
    )

    departure_station = (
        baris_data.stations_by_ids[flight['stationFrom']]
        if 'stationFrom' in flight and flight['stationFrom'] in baris_data.stations_by_ids
        else None
    )

    result = {
        'number': flight['title'],
        'scheduled': scheduled_dt.strftime('%Y-%m-%d %H:%M:%S %z'),
        'expected': expected_dt.strftime('%Y-%m-%d %H:%M:%S %z'),
        'status': _get_tablo_status(
            direction_type=query.direction_type, station=query.station, status=_event_update_status(flight_status)
        ),
        'title': _get_tablo_flight_title(baris_data, flight),
        'touch_url': url,
        'url': url,
        'arrival_settlement': _plane_station_title(arrival_station),
        'departure_settlement': _plane_station_title(departure_station),
    }

    if 'airlineID' in flight and flight['airlineID'] in baris_data.companies_by_ids:
        company = baris_data.companies_by_ids[flight['airlineID']]
        result['company'] = company.L_short_title() or company.L_title()

    terminal_name = terminal_update or flight['terminal']
    baggage_carousels = status['baggageCarousels']
    check_in_desks = status['checkInDesks']
    if terminal_name:
        result['terminal'] = terminal_name
    if baggage_carousels:
        result['baggage_carousels'] = status['baggageCarousels']
    if check_in_desks:
        result['check_in_desks'] = check_in_desks
    if gate:
        result['gate'] = gate

    return result


@traced_function
def format_suburban_station(data, query):
    translation.activate(query.language)

    station_urls = get_suburban_station_urls(direction_code='all', query=query)

    hide_all_direction_segments = ExperimentFlag.SUBURBAN_STATION_ALL_DIRECTION_SEGMENTS not in query.experiment_flags
    hide_arrival_direction = len(data.directions) > 1
    response = {
        'content': [
            dict({
                'segments': (
                    dump_suburban_station_segments(direction.segments[:5])
                    if not (hide_all_direction_segments and direction.type is DirectionType.ALL) else
                    None
                ),
                'title': _get_direction_title(direction),
                'total': direction.total,
                'type': (
                    'all' if direction.type is DirectionType.ALL else
                    'arrival' if direction.type is DirectionType.ARRIVAL else
                    'departure'
                ),
            }, **get_suburban_station_urls(direction_code=direction.code, query=query))
            for direction in data.directions
            if not (hide_arrival_direction and direction.type is DirectionType.ARRIVAL)
        ],
        'path_items': [dict(station_urls, **{'text': get_morda_host(query.tld)})],
        'type': 'suburban_directions',
    }
    response.update(station_urls)
    response.update(_get_suburban_station_texts(data, query))

    return response


def _get_service_urls(query):
    query_params = urlencode([get_tracking_query_item(TrackingParameter.PLANE_STATION)])
    return {
        'url': format_morda_url(path='/', query=query_params, tld=query.tld),
        'touch_url': format_touch_url(path='/', query=query_params, tld=query.tld)
    }


def _get_green_url_text(direction_type):
    if direction_type is DirectionType.DEPARTURE:
        return ugettext('flight_panel_departure')
    if direction_type is DirectionType.ARRIVAL:
        return ugettext('flight_panel_arrival')
    return ugettext('flight_panel')


@traced_function
def format_plane_station(baris_data, query):
    # type: (BarisData, PlaneStationQuery) -> Response

    translation.activate(query.language)

    service_urls = _get_service_urls(query)
    station_direction_urls = get_plane_station_urls(query=query, direction_type=query.direction_type)
    station_urls = get_plane_station_urls(query=query)

    segments = [
        _dump_baris_segment(flight=flight, baris_data=baris_data, query=query)
        for flight
        in baris_data.flights[:settings.PLANE_STATION_MAXIMUM_SEGMENTS]
        if 'datetime' in flight and flight['datetime']
    ]

    departure_content = dict({
            'selected': (query.direction_type is DirectionType.DEPARTURE),
            'segments': segments if query.direction_type is DirectionType.DEPARTURE else [],
            'title': ugettext('flight_departure'),
            'type': PLANE_DIRECTION_TYPE_TO_VALUE[DirectionType.DEPARTURE],
        },
        **get_plane_station_urls(query=query, direction_type=DirectionType.DEPARTURE)
    )

    arrival_content = dict({
            'selected': (query.direction_type is DirectionType.ARRIVAL),
            'segments': segments if query.direction_type is DirectionType.ARRIVAL else [],
            'title': ugettext('flight_arrival'),
            'type': PLANE_DIRECTION_TYPE_TO_VALUE[DirectionType.ARRIVAL],
        },
        **get_plane_station_urls(query=query, direction_type=DirectionType.ARRIVAL)
    )

    response = {
        'content': [
            # order is important
            departure_content,
            arrival_content
        ],
        'path_items': [
            dict(service_urls, **{'text': ugettext('ya_schedule')}),
            dict(station_urls, **{'text': query.station.L_title()}),
            dict(station_direction_urls, **{'text': _get_green_url_text(query.direction_type)}),
        ],
        'type': 'airport_panel_with_event',
        'search_props': {
            'plane-connection': 1,
        }
    }
    response.update(station_direction_urls)
    response.update(_get_plane_station_texts(query))

    return response
