# coding: utf-8
from __future__ import absolute_import, division, print_function, unicode_literals

import itertools
from datetime import datetime

from common.utils.date import UTC_TZ
from travel.library.python.tracing.instrumentation import traced_function
from travel.rasp.wizards.proxy_api.lib.segments.serialization import dump_segments, load_segments
from travel.rasp.wizards.proxy_api.lib.station.models import (
    EventUpdate, EventUpdateStatus, PlaneStationQuery, StationData, SuburbanDirection,
    SuburbanSegment, SuburbanStationQuery
)
from travel.rasp.wizards.proxy_api.lib.tracking import TrackingParameter, add_tracking_parameter
from travel.rasp.wizards.wizard_lib.protobuf_models.event_update_pb2 import TEventUpdate
from travel.rasp.wizards.wizard_lib.protobuf_models.station_direction_pb2 import TStationDirection
from travel.rasp.wizards.wizard_lib.protobuf_models.station_response_pb2 import TStationResponse
from travel.rasp.wizards.wizard_lib.station.direction_type import DirectionType
from travel.rasp.wizards.wizard_lib.serialization.date import parse_date
from travel.rasp.wizards.wizard_lib.serialization.language import parse_language


PLANE_DIRECTION_VALUE_TO_TYPE = {
    'departure': DirectionType.DEPARTURE,
    'arrival': DirectionType.ARRIVAL,
}
PLANE_DIRECTION_TYPE_TO_VALUE = {value: key for key, value in PLANE_DIRECTION_VALUE_TO_TYPE.items()}


def load_suburban_station_query(station, experiment_flags, query_params):
    return SuburbanStationQuery(
        station=station,
        event_date=parse_date(query_params.get('date'), station.pytz),
        language=parse_language(query_params.get('lang')),
        experiment_flags=experiment_flags,
        tld=query_params.get('tld'),
    )


def _parse_plane_direction_type(query_values):
    for query_value in query_values:
        direction_type = PLANE_DIRECTION_VALUE_TO_TYPE.get(query_value)
        if direction_type is not None:
            return direction_type


def load_plane_station_query(station, experiment_flags, query_params):
    return PlaneStationQuery(
        station=station,
        event_date=parse_date(query_params.get('date'), station.pytz),
        direction_type=_parse_plane_direction_type(query_params.getlist('query')),
        language=parse_language(query_params.get('lang')),
        experiment_flags=experiment_flags,
        tld=query_params.get('tld'),
    )


def load_suburban_station_data(raw_direction_data):
    proto_station = TStationResponse()
    proto_station.ParseFromString(raw_direction_data)
    return StationData(
        directions=tuple(
            SuburbanDirection(
                type=DirectionType[TStationDirection.EDirectionType.Name(proto_direction.Type)],
                code=proto_direction.SuburbanCode,
                total=proto_direction.Total,
                segments=load_segments(
                    proto_direction.Segments,
                    segment_factory=SuburbanSegment,
                    extra_kwargs_getter=lambda proto_segment: dict(
                        platform=proto_segment.TrainPlatform or None,
                        stops=proto_segment.TrainStops or None
                    )
                )
            )
            for proto_direction in proto_station.Directions
        )
    )


def _load_plane_segment_extra_fields(proto_segment, terminals):
    event_update_proto = proto_segment.EventUpdate if proto_segment.HasField('EventUpdate') else None
    return dict(
        arrival_terminal=terminals[proto_segment.ArrivalTerminalId] if proto_segment.ArrivalTerminalId else None,
        departure_terminal=terminals[proto_segment.DepartureTerminalId] if proto_segment.DepartureTerminalId else None,
        event_update=event_update_proto and EventUpdate(
            event_dt=datetime.utcfromtimestamp(event_update_proto.Timestamp).replace(tzinfo=UTC_TZ),
            terminal=event_update_proto.Terminal or None,
            gate=event_update_proto.Gate or None,
            status=EventUpdateStatus[TEventUpdate.EStatus.Name(event_update_proto.Status)],
            baggage_carousels=event_update_proto.BaggageCarousels or None,
            check_in_desks=event_update_proto.CheckInDesks or None,
        )
    )


@traced_function
def dump_suburban_station_segments(segments):
    result = dump_segments(segments)
    for segment, segment_dump in itertools.izip(segments, result):
        urls = segment.urls
        segment_dump.update({
            'arrival_station': segment.arrival_station.L_popular_title(),
            'departure_station': segment.departure_station.L_popular_title(),
            'platform': segment.platform,
            'stops': segment.stops,
            'touch_url': add_tracking_parameter(urls.mobile, TrackingParameter.SUBURBAN_STATION),
            'url': add_tracking_parameter(urls.desktop, TrackingParameter.SUBURBAN_STATION),
        })
    return result
