# coding: utf-8
from __future__ import absolute_import, division, print_function, unicode_literals

import itertools
import operator

from common.models.geo import Station
from common.precache.storage import BasePrecacheStorage
from travel.library.python.tracing.instrumentation import traced_function


def _get_major_station_id(stations):
    major_stations = tuple(itertools.groupby(stations, operator.itemgetter('majority_id')).next()[1])
    if len(major_stations) == 1:
        return major_stations[0]['id']


class SettlementStationsCache(BasePrecacheStorage):
    @classmethod
    def build_cache(cls):
        cache = {}

        for settlement_id, settlement_stations in itertools.groupby(
            Station.hidden_manager.filter(settlement__isnull=False)
                                  .order_by('settlement_id', 'majority_id')
                                  .values('id', 'settlement_id', 'majority_id', 't_type_id'),
            operator.itemgetter('settlement_id')
        ):
            settlement_stations = tuple(settlement_stations)

            major_station_id = _get_major_station_id(settlement_stations)
            if major_station_id is not None:
                cache[settlement_id, None] = major_station_id

            for t_type_id, t_type_stations in itertools.groupby(
                sorted(settlement_stations, key=operator.itemgetter('t_type_id', 'majority_id')),
                operator.itemgetter('t_type_id')
            ):
                major_station_id = _get_major_station_id(t_type_stations)
                if major_station_id is not None:
                    cache[settlement_id, t_type_id] = major_station_id

        return cache

    @classmethod
    @traced_function(name='proxy_api.lib.station.settlement_stations_cache.SettlementStationsCache.get')
    def get(cls, settlement, t_type_id=None):
        assert cls.is_precached()

        station_id = cls._cache.get((settlement.id, t_type_id))
        return Station.objects.get(pk=station_id) if station_id is not None else None
