# coding: utf-8
from __future__ import absolute_import, division, print_function, unicode_literals

import os
from os.path import realpath, dirname

from common.settings import *  # noqa: ImportStarUsed
from common.settings.configuration import BaseConfigurator, Configuration
from travel.rasp.library.python.common23.logging.qloud import context_extractor
from travel.rasp.wizards.proxy_api.settings.api_urls import *  # noqa: ImportStarUsed
from travel.rasp.wizards.wizard_lib.settings.hosts import TouchHostConfigurator, MordaHostConfigurator, TrainsHostConfigurator
from travel.rasp.wizards.wizard_lib.settings.locale import LocaleConfigurator


# настройки django

ADMINS = ()

ROOT_URLCONF = 'travel.rasp.wizards.proxy_api.urls'

MIDDLEWARE_CLASSES = (
    'django_opentracing.OpenTracingMiddleware',
    'travel.library.python.tracing.django.TracingMiddleware',
    'common.middleware.extract_context.ExtractRequestMiddleware',
    'common.middleware.ping.PingResponder',
)

INSTALLED_APPS = COMMON_INSTALLED_APPS + COMMON_CONTRIB_APPS + [  # noqa: ImportStarUsage
    'django_opentracing',
    'django.contrib.staticfiles',
    'raven.contrib.django.raven_compat',
    'rest_framework',
    'travel.rasp.wizards.proxy_api'
]

BASE_DIR = realpath(dirname(dirname(__file__)))

TEMPLATES = [{
    'BACKEND': 'django.template.backends.jinja2.Jinja2',
    'DIRS': [os.path.join(BASE_DIR, 'templates')],  # noqa: ImportStarUsage
    'OPTIONS': {
        'environment': 'travel.rasp.wizards.wizard_lib.utils.jinja2.get_environment',
        'extensions': ['jinja2.ext.i18n'],
        'lstrip_blocks': True,
        'trim_blocks': True,
    }
}]

# настройки third-party модулей

PYLOGCTX_REQUEST_EXTRACTOR = context_extractor

REST_FRAMEWORK = {
    'DEFAULT_RENDERER_CLASSES': ('travel.rasp.wizards.wizard_lib.utils.json_renderer.Renderer',)
}

OPENTRACING_TRACE_ALL = True
OPENTRACING_TRACED_ATTRIBUTES = ['path', 'method']

# настройки common

DISABLE_GROUP_CACHES = True

PRECACHE_ENABLE_FALLBACK_BY_DEFAULT = False

PRECACHE_MANAGERS = [
    'common.importinfo.models:Express2Country',
    'common.models.currency:Currency.objects',
    'common.models.geo:CodeSystem.objects',
    'common.models.geo:Country.objects',
    'common.models.geo:Direction.objects',
    'common.models.geo:DirectionFromTranslate.objects',
    'common.models.geo:District.objects',
    'common.models.geo:ExternalDirection.objects',
    'common.models.geo:Region.objects',
    'common.models.geo:ReplaceException',
    'common.models.geo:Settlement.objects',
    'common.models.geo:Station.code_manager',
    'common.models.geo:Station.objects',
    'common.models.geo:StationMajority.objects',
    'common.models.geo:StationTerminal.objects',
    'common.models.geo:StationType.objects',
    'common.models.schedule:Company.objects',
    'common.models.transport:TransportType.objects',
    'common.models.transport:TransportSubtype.objects',
    'geosearch.models:DefaultPoint.objects',
    'geosearch.models:NameSearchIndex',
    'geosearch.views.cache:ExternalDirectionCache',
    'geosearch.views.cache:StationCache',
    'geosearch.views.cache:SuburbanZoneCache',
    'travel.rasp.wizards.proxy_api.lib.station.settlement_stations_cache:SettlementStationsCache',
]

# настройки, зависящие от окружения

LOG_PATH = '/var/log/app'


class LoggingConfigurator(BaseConfigurator):
    def _configure_logging(self, settings, level='INFO'):
        log_format = '%(levelname)s %(asctime)s %(process)d %(name)s %(message)s'
        log_path = settings['LOG_PATH']

        settings['LOGGING'] = {
            'version': 1,
            'disable_existing_loggers': False,
            'filters': {
                'require_debug_true': {
                    '()': 'django.utils.log.RequireDebugTrue'
                },
                'context_filter': {
                    '()': 'travel.rasp.library.python.common23.logging.AddContextFilter'
                },
                'context_as_dict_filter': {
                    '()': 'travel.rasp.library.python.common23.logging.AddContextFilter',
                    'as_dict': True
                },
            },
            'formatters': {
                'default': {
                    '()': 'ylog.ExceptionFormatter',
                    'format': log_format,
                },
                'verbose': {
                    '()': 'ylog.ExceptionFormatter',
                    'format': log_format,
                    'full': True,
                    'show_locals': True,
                },
                'qloud_application': {
                    '()': 'travel.rasp.library.python.common23.logging.qloud.QloudJsonFormatter',
                    'tag': 'application'
                },
                'qloud_errors': {
                    '()': 'travel.rasp.library.python.common23.logging.qloud.QloudExceptionFormatter',
                    'add_request_info': True,
                    'tag': 'errors'
                },
            },
            'handlers': {
                'cout': {
                    'class': 'logging.StreamHandler',
                    'filters': ['require_debug_true'],
                    'formatter': 'default',
                },
                'application': {
                    'class': 'travel.rasp.library.python.common23.logging.WatchedFileHandler',
                    'filename': os.path.join(log_path, 'app.log'),
                    'formatter': 'default',
                },
                'db': {
                    'class': 'travel.rasp.library.python.common23.logging.WatchedFileHandler',
                    'filename': os.path.join(log_path, 'db.log'),
                    'formatter': 'default',
                },
                'traceback': {
                    'class': 'travel.rasp.library.python.common23.logging.WatchedFileHandler',
                    'filename': os.path.join(log_path, 'traceback.log'),
                    'formatter': 'verbose',
                },
                'qloud_application': {
                    'class': 'logging.StreamHandler',
                    'stream': 'ext://sys.stdout',
                    'filters': ['context_as_dict_filter'],
                    'formatter': 'qloud_application'
                },
                'qloud_errors': {
                    'class': 'logging.StreamHandler',
                    'stream': 'ext://sys.stderr',
                    'filters': ['context_as_dict_filter'],
                    'formatter': 'qloud_errors'
                },
            },
            'loggers': {
                '': {
                    'handlers': ['application', 'qloud_application'],
                    'level': level,
                },
                'common.db': {
                    'handlers': ['db'],
                    'level': 'DEBUG',
                    'propagate': False,
                },
                'django': {
                    'handlers': ['traceback', 'qloud_errors'],
                    'level': 'ERROR',
                    'propagate': False,
                },
                'django.db': {
                    'handlers': ['db'],
                    'level': 'DEBUG',
                    'propagate': False,
                },
                'proxy_api': {
                    'level': 'DEBUG'
                },
                'wizard_lib': {
                    'level': 'DEBUG'
                },
            }
        }
        settings['MYSQL_LOG_QUERIES'] = True

    def apply_development(self, settings):
        settings['LOG_PATH'] = os.path.join(settings['PROJECT_PATH'], 'log')
        self._configure_logging(settings, level='DEBUG')

    def apply_testing(self, settings):
        self._configure_logging(settings)

    def apply_stress(self, settings):
        self._configure_logging(settings)

    def apply_production(self, settings):
        self._configure_logging(settings)

if os.getenv('YANDEX_ENVIRONMENT_TYPE') == 'testing':
    YANDEX_DATA_CENTER = 'XXX'

Configuration.configurator_classes.extend((
    LocaleConfigurator, LoggingConfigurator, TouchHostConfigurator, MordaHostConfigurator, TrainsHostConfigurator
))

PROXY_API_REQUESTS_POOL_SIZE = 2  # number of threads in the pool should be equal to maximum number of parallel requests
PROXY_API_REQUEST_SOURCES_TIMEOUT = float(os.environ.get('PROXY_API_REQUEST_SOURCES_TIMEOUT', 0.09))

PLANE_STATION_MAXIMUM_SEGMENTS = 3  # maximum number or rows on plane station wizard

# timeout in wizards are very small, so timeout exception is ordinary situation and doesn't affect service perfomance
BARIS_BREAKER_PARAMS = {'fail_max': 50, 'reset_timeout': 10}
