# coding: utf-8
from __future__ import absolute_import, division, print_function, unicode_literals

from common.models_abstract.schedule import ExpressType
from common.models.currency import Price
from common.models.tariffs import AeroexTariff, TariffType
from common.precache.storage import BasePrecacheStorage


def _make_bidirectional_cache_key(departure_station_id, arrival_station_id, tariff_type_id):
    if departure_station_id > arrival_station_id:
        departure_station_id, arrival_station_id = arrival_station_id, departure_station_id
    return departure_station_id, arrival_station_id, tariff_type_id


class TariffsCache(BasePrecacheStorage):
    @classmethod
    def build_cache(cls):
        directional_cache, bidirectional_cache = {}, {}
        for (
            departure_station_id, arrival_station_id, price_value, currency, tariff_type_id,
            is_bidirectional, is_precalc
        ) in AeroexTariff.objects.filter(
            type__category=TariffType.USUAL_CATEGORY
        ).values_list(
            'station_from_id', 'station_to_id', 'tariff', 'currency', 'type_id', 'reverse', 'precalc'
        ).order_by('-precalc'):
            record = (is_precalc, price_value, currency)
            if is_bidirectional:
                bidirectional_cache[
                    _make_bidirectional_cache_key(departure_station_id, arrival_station_id, tariff_type_id)
                ] = record
            else:
                directional_cache[departure_station_id, arrival_station_id, tariff_type_id] = record
        return directional_cache, bidirectional_cache

    @classmethod
    def get_tariff(cls, raw_segment):
        directional_cache, bidirectional_cache = cls._cache

        thread = raw_segment.thread
        departure_station_id, arrival_station_id = raw_segment.departure_station.id, raw_segment.arrival_station.id
        tariff_type_id = thread.tariff_type_id or (
            TariffType.EXPRESS_ID if thread.express_type in (ExpressType.EXPRESS, ExpressType.AEROEXPRESS) else
            TariffType.DEFAULT_ID
        )

        records = filter(None, (
            directional_cache.get((departure_station_id, arrival_station_id, tariff_type_id)),
            bidirectional_cache.get(
                _make_bidirectional_cache_key(departure_station_id, arrival_station_id, tariff_type_id)
            ),
        ))
        if records:
            is_precalc, price_value, currency = min(records, key=lambda record: record[0])
            return Price(price_value, currency)
