# coding: utf-8
from __future__ import absolute_import, division, print_function, unicode_literals

import itertools

from django.utils.datastructures import MultiValueDict
from django.utils.encoding import force_bytes
from django.utils.http import urlencode
from django.utils.translation import ugettext

from common.utils.date import timedelta2minutes
from travel.rasp.wizards.suburban_wizard_api.lib.direction.filters import Filters
from travel.rasp.wizards.suburban_wizard_api.lib.serialization.points import parse_points
from travel.rasp.wizards.wizard_lib.direction.sorting import DEFAULT_SORTING
from travel.rasp.wizards.wizard_lib.serialization.date import parse_date
from travel.rasp.wizards.wizard_lib.serialization.direction import DirectionQuery
from travel.rasp.wizards.wizard_lib.serialization.experiment_flags import parse_experiment_flags
from travel.rasp.wizards.wizard_lib.serialization.language import parse_language
from travel.rasp.wizards.wizard_lib.serialization.price import dump_price
from travel.rasp.wizards.wizard_lib.serialization.segments import dump_duration, get_minimum_duration
from travel.rasp.wizards.wizard_lib.views.helpers.service_urls import format_morda_url, format_touch_url


SEGMENTS_LIMIT = 5


def _dump_datetime(local_dt):
    return local_dt.strftime('%Y-%m-%d %H:%M:%S %z')


def _dump_minimum_price(segment_prices):
    segment_prices = filter(None, segment_prices)
    if segment_prices:
        return dump_price(min(segment_prices))


def _dump_segment(segment, language, tld):
    departure_station, arrival_station = segment.departure_station, segment.arrival_station

    url_path = b'/thread/{}/'.format(force_bytes(segment.thread_uid))
    url_query = urlencode((
        ('departure', segment.thread_start_date),
        ('station_from', departure_station.id),
        ('station_to', arrival_station.id),
    ))

    return {
        'arrival': _dump_datetime(segment.arrival_local_dt),
        'departure': _dump_datetime(segment.departure_local_dt),
        'duration': timedelta2minutes(segment.arrival_local_dt - segment.departure_local_dt),
        'from_station': ugettext('from_station') % {
            'station_title': departure_station.L_popular_title(case='genitive')
        },
        'number': segment.train_number,
        'title': getattr(segment.train_title, language),
        'touch_url': format_touch_url(url_path, url_query, tld),
        'url': format_morda_url(url_path, url_query, tld),
    }


def dump_segments(segments, found_departure_date, language, tld):
    total_segments = len(segments)
    segments = segments[:SEGMENTS_LIMIT]
    segment_prices = tuple(segment.price for segment in segments)

    return {
        'departure_date': found_departure_date.strftime('%Y-%m-%d'),
        'minimum_duration': dump_duration(get_minimum_duration(segments)),
        'minimum_price': _dump_minimum_price(segment_prices),
        'segments': [
            dict(_dump_segment(segment, language, tld), **{'price': dump_price(segment_price)})
            for segment, segment_price in itertools.izip(segments, segment_prices)
        ],
        'total': total_segments,
        'transport': 'suburban',
    }


def load_query(query_params):
    departure_point, arrival_point = parse_points(query_params, 'point', 'point_to')
    main_reqid = query_params.get('main_reqid') or None

    return DirectionQuery(
        departure_point=departure_point,
        arrival_point=arrival_point,
        original_departure_point=None,
        original_arrival_point=None,
        departure_date=parse_date(query_params.get('date'), departure_point.pytz),
        filters=Filters.load(MultiValueDict()),
        language=parse_language(query_params.get('lang')),
        experiment_flags=parse_experiment_flags(query_params.get('exp_flags')),
        sorting=DEFAULT_SORTING,
        tld=query_params.get('tld'),
        limit=None,
        main_reqid=main_reqid,
    )
