# coding: utf-8
from __future__ import absolute_import, division, print_function, unicode_literals

from rest_framework import serializers

from geosearch.views.point import PointSearchError, SuburbanPointSearch
from geosearch.views.pointlist import PointList
from geosearch.views.pointtopoint import process_points_lists, SamePointError
from travel.rasp.wizards.wizard_lib.serialization.point import parse_point


def _parse_pointlist(query_params, point_key_arg, settlement_geoid_arg, point_name_arg):
    point = parse_point(query_params, point_key_arg, settlement_geoid_arg, allow_blank=True)
    point_name = query_params.get(point_name_arg)
    if point is not None:
        if point_name:
            raise serializers.ValidationError(
                'only one of arguments have to be specified: {}, {} or {}'
                .format(point_key_arg, settlement_geoid_arg, point_name_arg)
            )
        return PointList(point, exact_variant=True)

    if not point_name:
        raise serializers.ValidationError(
            'one of arguments have to be specified: {}, {} or {}'
            .format(point_key_arg, settlement_geoid_arg, point_name_arg)
        )

    try:
        return SuburbanPointSearch.find_point(point_name)
    except PointSearchError as error:
        raise serializers.ValidationError('invalid point name {}: {}'.format(point_name_arg, error.message))


def parse_points(
    query_params, departure_point_name_arg='departure_point_name', arrival_point_name_arg='arrival_point_name'
):
    departure_pointlist = _parse_pointlist(
        query_params, 'departure_point_key', 'departure_settlement_geoid', departure_point_name_arg
    )
    arrival_pointlist = _parse_pointlist(
        query_params, 'arrival_point_key', 'arrival_settlement_geoid', arrival_point_name_arg
    )
    try:
        departure_pointlist, arrival_pointlist = process_points_lists(
            departure_pointlist, arrival_pointlist, disable_replace=True, suburban=True
        )
    except SamePointError:
        raise serializers.ValidationError('departure and arrival points are same')

    return departure_pointlist.point, arrival_pointlist.point
