# coding: utf-8
from __future__ import absolute_import, division, print_function, unicode_literals

import calendar
from collections import namedtuple

from django.utils.encoding import force_bytes
from django.utils.http import urlencode
from rest_framework import serializers

from common.models.geo import Station
from travel.library.python.tracing.instrumentation import traced_function
from travel.rasp.wizards.wizard_lib.protobuf_models.station_response_pb2 import TStationResponse
from travel.rasp.wizards.wizard_lib.protobuf_models.station_direction_pb2 import TStationDirection
from travel.rasp.wizards.wizard_lib.protobuf_models.thread_pb2 import TThread
from travel.rasp.wizards.wizard_lib.serialization.date import parse_date
from travel.rasp.wizards.wizard_lib.serialization.experiment_flags import parse_experiment_flags
from travel.rasp.wizards.wizard_lib.serialization.language import parse_language
from travel.rasp.wizards.wizard_lib.serialization.thread_express_type import ThreadExpressType
from travel.rasp.wizards.wizard_lib.views.helpers.service_urls import format_morda_url, format_touch_url

StationQuery = namedtuple('StationQuery', 'station, event_date, language, tld, experiment_flags, main_reqid')


def _parse_station_id(station_id):
    if station_id is None:
        raise serializers.ValidationError('station_id is required')

    try:
        return Station.hidden_manager.get(id=station_id)
    except Station.DoesNotExist:
        raise serializers.ValidationError('unknown station_id')


@traced_function
def load_query(query_params):
    station = _parse_station_id(query_params.get('station_id'))
    main_reqid = query_params.get('main_reqid') or None

    return StationQuery(
        station=station,
        event_date=parse_date(query_params.get('event_date'), station.pytz),
        language=parse_language(query_params.get('lang')),
        tld=query_params.get('tld'),
        experiment_flags=parse_experiment_flags(query_params.get('exp_flags')),
        main_reqid=main_reqid,
    )


def _update_timestamp(proto_timestamp, local_dt):
    proto_timestamp.Timestamp = calendar.timegm(local_dt.utctimetuple())
    proto_timestamp.UtcOffset = int(local_dt.utcoffset().total_seconds())


def _update_thread(proto_thread, thread, thread_start_date, language):
    proto_thread.Number = thread.number
    proto_thread.StartDate = thread_start_date.toordinal()
    proto_thread.Title = getattr(thread.title, language)

    if thread.t_subtype_id is not None:
        proto_thread.TransportSubtypeId = thread.t_subtype_id

    if thread.express_type:
        thread_express_type = ThreadExpressType(thread.express_type)
        proto_thread.ExpressType = TThread.EExpressType.Value(thread_express_type.name)


def _update_segment_urls(proto_urls, raw_segment, thread_start_date, query):
    url_path = b'/thread/{}/'.format(force_bytes(raw_segment.thread.uid))
    parameters = [
        ('departure', thread_start_date),
        ('station_from', raw_segment.departure_station.id),
        ('station_to', raw_segment.arrival_station.id)
    ]
    if query.main_reqid is not None:
        parameters.append(('wizardReqId', query.main_reqid))
    url_query = urlencode(parameters)
    proto_urls.Desktop = format_morda_url(url_path, url_query, query.tld)
    proto_urls.Mobile = format_touch_url(url_path, url_query, query.tld)


@traced_function
def dump_station_response(raw_directions, query, segments_limit=None):
    proto_station = TStationResponse()

    for direction, direction_count, direction_segments in raw_directions:
        proto_direction = proto_station.Directions.add(
            SuburbanCode=direction.code,
            Total=direction_count,
            Type=TStationDirection.EDirectionType.Value(direction.type.name),
        )

        for raw_segment in direction_segments[:segments_limit]:
            thread = raw_segment.thread
            thread_start_date = raw_segment.thread_start_dt.date()
            departure_dt = raw_segment.departure_dt
            arrival_dt = raw_segment.arrival_dt
            departure_station = raw_segment.departure_station
            arrival_station = raw_segment.arrival_station

            proto_segment = proto_direction.Segments.add(
                DepartureStationId=departure_station.id,
                ArrivalStationId=arrival_station.id,
                TrainPlatform=raw_segment.event_stop.platform,
                TrainStops=getattr(raw_segment.event_stop.stops_text, query.language, None)
            )

            _update_timestamp(proto_segment.DepartureTimestamp, departure_dt.astimezone(departure_station.pytz))
            _update_timestamp(proto_segment.ArrivalTimestamp, arrival_dt.astimezone(arrival_station.pytz))
            _update_thread(proto_segment.Thread, thread, thread_start_date, query.language)
            _update_segment_urls(proto_segment.Urls, raw_segment, thread_start_date, query)

    return proto_station
