# coding: utf-8
from __future__ import absolute_import, division, print_function, unicode_literals

from collections import defaultdict

from travel.rasp.library.python.common23.date import environment
from common.utils.date import MSK_TZ
from travel.library.python.tracing.instrumentation import traced_function
from travel.rasp.wizards.suburban_wizard_api.lib.schedule_cache import schedule_cache
from travel.rasp.wizards.suburban_wizard_api.lib.station.suburban_directions_cache import ALL_DIRECTION, SuburbanDirectionsCache
from travel.rasp.wizards.wizard_lib.event_date_query import make_fixed_date_query


def find_date_segments(station, event_date):
    event_date_query = make_fixed_date_query(event_date, station.pytz)
    return event_date_query.raw_filter_func(
        schedule_cache.find_station_segments(station, event_date_query.min_event_dt)
    )


def find_future_segments(station):
    return schedule_cache.find_station_segments(station, environment.now_aware())


def find_hacked_future_segments(station):
    hacked_now = MSK_TZ.localize(environment.now_aware().astimezone(station.pytz).replace(tzinfo=None))
    return schedule_cache.find_station_segments(station, hacked_now)


def split_segments_by_directions(raw_segments_iter, directions):
    direction_event_dates = dict.fromkeys(directions)
    segments_by_direction = defaultdict(list)

    for raw_segment, segment_direction in SuburbanDirectionsCache.iter_raw_segments_with_directions(raw_segments_iter):
        event_dt = raw_segment.event_dt
        event_station = raw_segment.event_station
        event_date = event_dt.astimezone(event_station.pytz).date()

        for direction in {segment_direction, ALL_DIRECTION}:
            try:
                direction_event_date = direction_event_dates[direction]
            except KeyError:
                continue

            if direction_event_date is None:
                direction_event_dates[direction] = event_date
            elif direction_event_date != event_date:
                del direction_event_dates[direction]
                if direction_event_dates:
                    continue
                break

            segments_by_direction[direction].append(raw_segment)
        else:
            continue
        break

    return dict(segments_by_direction)


@traced_function
def find_directions_with_segments(station, event_date=None, hacked_future=False):
    directions_with_counts = SuburbanDirectionsCache.list_directions_with_counts(station)
    if not directions_with_counts:
        return ()

    segments_by_direction = split_segments_by_directions(
        directions=zip(*directions_with_counts)[0],
        raw_segments_iter=(
            find_date_segments(station, event_date) if event_date is not None else
            find_hacked_future_segments(station) if hacked_future else
            find_future_segments(station)
        )
    )
    return tuple(
        (direction, direction_count, segments_by_direction[direction])
        for direction, direction_count in directions_with_counts
        if direction in segments_by_direction
    )
