# coding=utf-8
from __future__ import absolute_import, division, print_function, unicode_literals

import os
from os.path import realpath, dirname

from common.settings import *  # noqa: ImportStarUsed
from common.settings.configuration import BaseConfigurator, Configuration
from travel.rasp.library.python.common23.logging.configure import configure_default_logging
from travel.rasp.library.python.common23.logging.qloud import context_extractor
from travel.rasp.wizards.wizard_lib.settings.hosts import TouchHostConfigurator, MordaHostConfigurator
from travel.rasp.wizards.wizard_lib.settings.locale import LocaleConfigurator


# настройки django

ADMINS = ()

ROOT_URLCONF = 'travel.rasp.wizards.suburban_wizard_api.urls'

MIDDLEWARE_CLASSES = (
    'django_opentracing.OpenTracingMiddleware',
    'travel.library.python.tracing.django.TracingMiddleware',
    'common.middleware.extract_context.ExtractRequestMiddleware',
    'common.middleware.ping.PingResponder',
)

INSTALLED_APPS = COMMON_INSTALLED_APPS + COMMON_CONTRIB_APPS + [  # noqa: ImportStarUsage
    'django_opentracing',
    'django.contrib.staticfiles',
    'raven.contrib.django.raven_compat',
    'rest_framework',
    'travel.rasp.wizards.suburban_wizard_api'
]

TEMPLATES = [{
    'BACKEND': 'django.template.backends.jinja2.Jinja2',
    'DIRS': [realpath(dirname(dirname(__file__)))],  # noqa: ImportStarUsage
    'OPTIONS': {
        'environment': 'travel.rasp.wizards.wizard_lib.utils.jinja2.get_environment',
        'extensions': ['jinja2.ext.i18n'],
        'lstrip_blocks': True,
        'trim_blocks': True,
    }
}]

# настройки third-party модулей

PYLOGCTX_REQUEST_EXTRACTOR = context_extractor

REST_FRAMEWORK = {
    'DEFAULT_RENDERER_CLASSES': ('travel.rasp.wizards.wizard_lib.utils.json_renderer.Renderer',)
}

OPENTRACING_TRACE_ALL = True
OPENTRACING_TRACED_ATTRIBUTES = ['path', 'method']

# настройки common

DISABLE_GROUP_CACHES = True

PRECACHE_ENABLE_FALLBACK_BY_DEFAULT = False

PRECACHE_MANAGERS = [
    'common.importinfo.models:Express2Country',
    'common.models.currency:Currency.objects',
    'common.models.geo:CodeSystem.objects',
    'common.models.geo:Country.objects',
    'common.models.geo:District.objects',
    'common.models.geo:ExternalDirection.objects',
    'common.models.geo:Region.objects',
    'common.models.geo:Settlement.objects',
    'common.models.geo:Station.code_manager',
    'common.models.geo:Station.objects',
    'common.models.geo:StationMajority.objects',
    'common.models.transport:TransportType.objects',
    'geosearch.models:DefaultPoint.objects',
    'geosearch.views.cache:ExternalDirectionCache',
    'geosearch.views.cache:StationCache',
    'travel.rasp.wizards.suburban_wizard_api.lib.schedule_cache:schedule_cache',
    'travel.rasp.wizards.suburban_wizard_api.lib.direction.tariffs_cache:TariffsCache',
    'travel.rasp.wizards.suburban_wizard_api.lib.station.suburban_directions_cache:SuburbanDirectionsCache',
    'travel.rasp.wizards.wizard_lib.utils.namesearchindex_precache_suburban:PrecacheManager',
]

# настройки, зависящие от окружения

LOG_PATH = '/var/log/app'


class LoggingConfigurator(BaseConfigurator):
    def _configure_logging(self, settings, level='INFO'):
        log_format = '%(levelname)s %(asctime)s %(process)d %(name)s %(message)s'
        log_path = settings['LOG_PATH']

        settings['LOGGING'] = configure_default_logging(settings, line_format=log_format)

        settings['LOGGING']['filters'].update({
            'require_debug_true': {
                '()': 'django.utils.log.RequireDebugTrue'
            }
        })

        settings['LOGGING']['formatters'].update({
            'default': {
                '()': 'ylog.ExceptionFormatter',
                'format': log_format,
            },
            'json': {
                'format': '%(message)s',
                '()': 'pythonjsonlogger.jsonlogger.JsonFormatter'
            }
        })

        settings['LOGGING']['handlers'].update({
            'cout': {
                'class': 'logging.StreamHandler',
                'filters': ['require_debug_true'],
                'formatter': 'default',
            },
            'db': {
                'class': 'travel.rasp.library.python.common23.logging.WatchedFileHandler',
                'filename': os.path.join(log_path, 'db.log'),
                'formatter': 'default',
            },
            'yt.direction': {
                'class': 'travel.rasp.library.python.common23.logging.WatchedFileHandler',
                'filename': os.path.join(log_path, 'yt', 'direction.log'),
                'formatter': 'json',
                'level': 'INFO',
            }
        })

        settings['LOGGING']['loggers'].update({
            'common.db': {
                'handlers': ['db'],
                'level': level,
                'propagate': False,
            },
            'django.db': {
                'handlers': ['db'],
                'level': 'DEBUG',
                'propagate': False,
            },
            'kikimr': {
                'handlers': ['exception.log', 'deploy_stderr'],
                'level': 'ERROR',
                'propagate': False,
            },
            'suburban_wizard_api': {
                'handlers': ['cout'],
                'level': level
            },
            'wizard_lib': {
                'handlers': ['cout'],
                'level': level
            },
            'yt.direction': {
                'handlers': ['yt.direction'],
                'level': 'INFO',
                'propagate': False,
            }
        })

    def apply_development(self, settings):
        settings['LOG_PATH'] = os.path.join(settings['PROJECT_PATH'], 'log')
        self._configure_logging(settings, level='DEBUG')

    def apply_testing(self, settings):
        self._configure_logging(settings, level='DEBUG')

    def apply_stress(self, settings):
        self._configure_logging(settings)

    def apply_production(self, settings):
        self._configure_logging(settings)

if os.getenv('YANDEX_ENVIRONMENT_TYPE') == 'testing':
    YANDEX_DATA_CENTER = 'XXX'

Configuration.configurator_classes.extend((
    LoggingConfigurator, LocaleConfigurator, TouchHostConfigurator, MordaHostConfigurator
))
Configuration().apply(globals())
