# coding: utf-8
from __future__ import absolute_import, division, print_function, unicode_literals

import logging

from django.utils import translation
from django.utils.http import urlencode
from rest_framework import status
from rest_framework.decorators import api_view
from rest_framework.response import Response

from travel.library.python.tracing.django import traced_view
from travel.library.python.tracing.instrumentation import child_span
from travel.rasp.wizards.suburban_wizard_api.lib.direction.logger import suburban_direction_logger
from travel.rasp.wizards.suburban_wizard_api.lib.direction.segments import join_variants, make_segments, split_segments
from travel.rasp.wizards.suburban_wizard_api.lib.schedule_cache import schedule_cache
from travel.rasp.wizards.suburban_wizard_api.lib.serialization.direction import dump_segment, load_query
from travel.rasp.wizards.wizard_lib.direction.segments import find_schedule_segments
from travel.rasp.wizards.wizard_lib.event_date_query import make_event_date_query
from travel.rasp.wizards.wizard_lib.serialization.direction import dump_query
from travel.rasp.wizards.wizard_lib.utils.functional import compose
from travel.rasp.wizards.wizard_lib.utils.yasm_metric_manager import YasmMetricManager
from travel.rasp.wizards.wizard_lib.views.helpers.error_handlers import log_validation_error
from travel.rasp.wizards.wizard_lib.views.helpers.service_urls import format_touch_url, format_morda_url

logger = logging.getLogger(__name__)
yasm_metric_manager = YasmMetricManager(__name__, logger)


@traced_view()
@api_view()
@log_validation_error
@suburban_direction_logger.decorate
def direction_view(request, log_context):
    yasm_metric_manager.send_one('request_cnt')

    query = load_query(request.GET)
    translation.activate(query.language)
    log_context.store_query(query)

    departure_date_query = make_event_date_query(query.departure_date, query.departure_point.pytz)
    found_departure_date, raw_segments = find_schedule_segments(
        schedule_cache, query.departure_point, query.arrival_point, departure_date_query,
    )
    if not raw_segments:
        yasm_metric_manager.send_one('no_data_cnt')
        return Response(status=status.HTTP_204_NO_CONTENT)

    segments = make_segments(raw_segments)
    log_context.store_segments(segments)

    segments = compose(split_segments, query.filters.apply, query.sorting.apply, join_variants)(segments)
    log_context.store_filtered_segments(segments)

    search_url_query = urlencode({
        'fromName': query.departure_point.L_title(),
        'fromId': query.departure_point.point_key,
        'toName': query.arrival_point.L_title(),
        'toId': query.arrival_point.point_key,
        'when': found_departure_date.isoformat()
    })

    yasm_metric_manager.send_one('with_data_cnt')

    with child_span('suburban-wizard-api.views.direction.direction_view::Response.__init__'):
        return Response(dict(dump_query(query), **{
            'search_touch_url': format_touch_url('/search/suburban/', search_url_query, query.tld),
            'search_url': format_morda_url('/search/suburban/', search_url_query, query.tld),
            'segments': [
                dump_segment(segment, query.language)
                for segment in segments
            ]
        }))
