# coding: utf-8
from __future__ import absolute_import, division, print_function, unicode_literals

from datetime import timedelta

from django.utils import translation
from django.utils.http import urlencode
from rest_framework import status
from rest_framework.decorators import api_view
from rest_framework.response import Response

from travel.rasp.library.python.common23.date import environment
from travel.rasp.wizards.suburban_wizard_api.lib.direction.logger import suburban_direction_logger
from travel.rasp.wizards.suburban_wizard_api.lib.direction.segments import join_variants, make_next_query, make_segments, split_segments
from travel.rasp.wizards.suburban_wizard_api.lib.schedule_cache import schedule_cache
from travel.rasp.wizards.suburban_wizard_api.lib.serialization.legacy_direction import dump_segments, load_query
from travel.rasp.wizards.wizard_lib.direction.segments import find_schedule_segments
from travel.rasp.wizards.wizard_lib.serialization.legacy_direction import dump_query
from travel.rasp.wizards.wizard_lib.utils.functional import compose
from travel.rasp.wizards.wizard_lib.utils.text import render_texts
from travel.rasp.wizards.wizard_lib.views.helpers.error_handlers import log_validation_error
from travel.rasp.wizards.wizard_lib.views.helpers.service_urls import format_morda_url, format_touch_url, get_morda_host


def _get_texts(departure_point, arrival_point, segments, experiment_flags):
    segment_prices = filter(None, (segment.price for segment in segments))
    context = {
        'departure_point': departure_point,
        'arrival_point': arrival_point,
        'today': environment.today(),
        'minimum_duration': min(segment.arrival_local_dt - segment.departure_local_dt for segment in segments),
        'minimum_price': min(segment_prices) if segment_prices else None
    }
    return render_texts('lib/direction/texts', context, experiment_flags)


def _get_tomorrow_date(local_tz):
    return (environment.now_aware() + timedelta(days=1)).astimezone(local_tz).date()


def _get_links(url_query, local_tz, tld):
    tomorrow_url_query = b'{query}&when={when}'.format(query=url_query, when=_get_tomorrow_date(local_tz))
    return {
        'next_link': {
            'title': translation.ugettext('next_link_title'),
            'touch_url': format_touch_url('/search/suburban/next/', url_query, tld),
            'url': format_morda_url('/search/suburban/next/', url_query, tld),
        },
        'tomorrow_link': {
            'title': translation.ugettext('tomorrow_link_title'),
            'touch_url': format_touch_url('/search/suburban/', tomorrow_url_query, tld),
            'url': format_morda_url('/search/suburban/', tomorrow_url_query, tld),
        },
    }


def _get_search_urls(url_query, found_departure_date, tld):
    search_url_query = b'{query}&when={when}'.format(query=url_query, when=found_departure_date)
    return {
        'url': format_morda_url('/search/suburban/', search_url_query, tld),
        'touch_url': format_touch_url('/search/suburban/', search_url_query, tld),
    }


@api_view()
@log_validation_error
@suburban_direction_logger.decorate
def legacy_direction_view(request, log_context):
    query = load_query(request.query_params)
    translation.activate(query.language)
    log_context.store_query(query)

    departure_date_query = make_next_query(query.departure_date, query.departure_point.pytz)
    found_departure_date, raw_segments = find_schedule_segments(
        schedule_cache, query.departure_point, query.arrival_point, departure_date_query
    )
    if not raw_segments:
        return Response(status=status.HTTP_204_NO_CONTENT)

    segments = make_segments(raw_segments)
    log_context.store_segments(segments)

    segments = compose(split_segments, query.filters.apply, join_variants)(segments)
    log_context.store_filtered_segments(segments)

    if not segments:
        return Response(status=status.HTTP_204_NO_CONTENT)

    direction_url_query = urlencode((
        ('fromName', query.departure_point.L_title()),
        ('fromId', query.departure_point.point_key),
        ('toName', query.arrival_point.L_title()),
        ('toId', query.arrival_point.point_key)
    ))
    search_urls = _get_search_urls(direction_url_query, found_departure_date, query.tld)

    response = {
        'type': 'transports_with_default',
        'query': dump_query(query),
        'path_items': [dict(search_urls, **{'text': get_morda_host(query.tld)})],
        'default_transport': dict(
            dump_segments(segments, found_departure_date, query.language, query.tld),
            **search_urls
        ),
    }
    response.update(search_urls)
    response.update(_get_texts(
        query.departure_point,
        query.arrival_point,
        segments,
        query.experiment_flags,
    ))
    response.update(_get_links(direction_url_query, query.departure_point.pytz, query.tld))

    return Response(response)
