# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import itertools

from common.utils.text import normalize
from travel.rasp.wizards.wizard_lib.direction.filters import BaseFilter
from travel.rasp.wizards.wizard_lib.serialization.price import dump_price


# RASPWIZARDS-704. Hardcoded plural form for delux trains
DELUX_TRAIN_PLURAL_FORM = {
    257: 'Поезда Аллегро',
    140: 'Гранд Экспрессы',
    294: 'Демидовские экспрессы',
    85: 'Поезда Красная стрела',
    252: 'Ласточки',
    228: 'Латвия экспрессы',
    207: 'Невские экспрессы',
    109: 'Поезда Полярная стрела',
    258: 'Сапсаны',
    288: 'Поезда Северная Пальмира',
    95: 'Славянские экспрессы',
    117: 'Поезда Смена А. Бетанкур',
    260: 'Стрижи',

    # tests
    320: 'some_train',
    50111: 'Сапсаны',
}

DELUX_TRAIN_NAME_FORMS_TO_ID = {
    'гранд-экспресс': 140,
    'гранд экспресс': 140,
    'гранд-экспрессы': 140,
    'гранд экспрессы': 140,
    'ласточка': 252,
    'ласточки': 252,
    'сапсаны': 258,
    'стриж': 260,
    'стрижи': 260,
    'cлавянский экспресс': 95,
    'cлавянские экспрессы': 95,
    'латвия экспрессы': 228,
    'латвия экспресс': 228,
    'невские экспрессы': 207,
    'невский экспресс': 207,
    'демидовский экспресс': 294,
    'демидовские экспрессы': 294,

    # tests
    'сапсан-тест': 50111,
}


def _get_brand_values(brand_id, brand_title):
    return {(True, brand_id), (False, normalize(brand_title))}


class BrandFilter(BaseFilter):
    def __init__(self, selected):
        self._selected = selected
        self._brands = {}
        self._minimum_prices = {}
        super(BrandFilter, self).__init__(selected)

    @classmethod
    def load(cls, values):
        selected = set()

        for value in values:
            try:
                brand_id = int(value)
            except ValueError:
                normalized = normalize(value)

                if normalized in DELUX_TRAIN_NAME_FORMS_TO_ID:
                    selected.add((True, DELUX_TRAIN_NAME_FORMS_TO_ID[normalized]))
                else:
                    selected.add((False, normalized))
            else:
                selected.add((True, brand_id))

        return cls(selected)

    def bind(self, variants):
        brands = self._brands
        minimum_prices = self._minimum_prices

        variant_values = []
        for variant in variants:
            brand = variant.segment.train_brand
            if brand is None:
                variant_values.append(set())
            else:
                brand_id = brand.id
                if brand_id in brands:
                    brand_title = brands[brand_id]['title']
                else:
                    brand_title = brand.L_title()
                    brands[brand_id] = {'title': brand_title, 'high_speed': brand.high_speed}

                places_group = variant.places_group
                if places_group is not None:
                    price = places_group.price
                    if brand_id not in minimum_prices or price < minimum_prices[brand_id]:
                        minimum_prices[brand_id] = price

                variant_values.append(_get_brand_values(brand_id, brand_title))

        self._variant_values = variant_values

    def update_availability(self, selectors):
        assert self.is_bound
        available_values = tuple(itertools.compress(self._variant_values, selectors))
        self._available = set.union(*available_values) if available_values else set()

    def list_selectors(self):
        assert self.is_bound
        selected, variant_values = self._selected, self._variant_values

        selectors = [bool(variant_value & selected) for variant_value in variant_values]
        return selectors if any(selectors) else [True] * len(variant_values)

    def dump(self):
        selected, available, brands, minimum_prices = \
            self._selected, self._available, self._brands, self._minimum_prices

        result = []
        for brand_id in sorted(brands):
            brand_title = brands[brand_id]['title']
            brand_high_speed = brands[brand_id]['high_speed']
            brand_values = _get_brand_values(brand_id, brand_title)
            result.append({
                'value': brand_id,
                'title': brand_title,
                'selected': bool(brand_values & selected),
                'available': bool(brand_values & available),
                'is_high_speed': brand_high_speed,
                'minimum_price': dump_price(minimum_prices.get(brand_id))
            })

        return result

    def get_brand_title(self):
        selected, brands = self._selected, self._brands
        selected_brands_count = 0
        last_selected_id = None
        for brand_id in brands:
            brand_title = brands[brand_id]['title']
            brand_values = _get_brand_values(brand_id, brand_title)
            if bool(brand_values & selected):
                selected_brands_count += 1
                last_selected_id = brand_id
        if selected_brands_count == 1:
            # TODO RASPWIZARDS-704. Remove use hardcoded plural form for delux trains
            try:
                return True, DELUX_TRAIN_PLURAL_FORM[last_selected_id]
            except KeyError:
                return False, None
        else:
            return False, None

    def get_search_params(self):
        assert self.is_bound
        selected, brands = self._selected, self._brands
        return tuple(
            ('highSpeedTrain', brand_id)
            for brand_id in brands
            if bool(_get_brand_values(brand_id, brands[brand_id]['title']) & selected)
        ) if selected and brands else ()
