# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

from rest_framework import serializers

from common.apps.train_order.enums import CoachType
from travel.rasp.wizards.wizard_lib.direction.filters import BaseFilter
from travel.rasp.wizards.wizard_lib.serialization.price import dump_price


class CoachTypeFilter(BaseFilter):
    VALUES = {t.value for t in CoachType}

    def __init__(self, selected):
        self._selected = selected
        self._minimum_prices = {}
        super(CoachTypeFilter, self).__init__(self._selected)

    @classmethod
    def load(cls, values):
        if not cls.VALUES.issuperset(values):
            raise serializers.ValidationError('invalid coach filter value: values should be one of [{}]'.format(
                ', '.join(sorted(cls.VALUES)))
            )

        return cls(frozenset(values))

    def bind(self, variants):
        minimum_prices = self._minimum_prices

        variant_values = []
        for variant in variants:
            value = None
            places_group = variant.places_group
            if places_group is not None and places_group.coach_type in self.VALUES:
                value = places_group.coach_type
                price = places_group.price
                if value not in minimum_prices or price < minimum_prices[value]:
                    minimum_prices[value] = price
            variant_values.append(value)

        self._variant_values = variant_values

    def list_selectors(self):
        assert self.is_bound
        selected, variant_values = self._selected, self._variant_values

        if not selected:
            return [True] * len(variant_values)

        return [variant_value in selected for variant_value in variant_values]

    def dump(self):
        selected, available, minimum_prices = self._selected, self._available, self._minimum_prices
        return [
            {
                'value': value,
                'selected': value in selected,
                'available': value in available,
                'minimum_price': dump_price(minimum_prices.get(value))
            }
            for value in sorted(self.VALUES)
        ]

    def get_search_params(self):
        return tuple(('trainTariffClass', value) for value in self._selected) if self._selected else ()
