# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import itertools

from rest_framework import serializers

from travel.rasp.wizards.train_wizard_api.lib.facility_provider import facility_provider
from travel.rasp.wizards.wizard_lib.direction.filters import BaseFilter


class FacilityFilter(BaseFilter):
    def __init__(self, selected):
        self._selected = selected
        super(FacilityFilter, self).__init__(self._selected)

    @classmethod
    def load(cls, values):
        all_codes = facility_provider.get_all_codes()
        if not all_codes.issuperset(values):
            raise serializers.ValidationError('invalid facility filter value: values should be one of [{}]'.format(
                ', '.join(sorted(all_codes)))
            )

        return cls(frozenset(values))

    def bind(self, variants):
        variant_values = []
        for variant in variants:
            variant_values.append(frozenset(filter(None, map(
                facility_provider.get_code_by,
                variant.segment.facilities_ids or ()
            ))))
        self._variant_values = variant_values

    def list_selectors(self):
        assert self.is_bound
        selected, variant_values = self._selected, self._variant_values

        if not selected:
            return [True] * len(variant_values)

        return [selected.issubset(variant_value) for variant_value in variant_values]

    def dump(self):
        selected, available = self._selected, self._available

        return [
            {
                'value': value,
                'selected': value in selected,
                'available': value in available
            }
            for value in sorted(facility_provider.get_all_codes())
        ]

    def update_availability(self, selectors):
        assert self.is_bound
        for values in itertools.compress(self._variant_values, selectors):
            self._available |= values
