# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import bisect

from rest_framework import serializers

from travel.rasp.wizards.wizard_lib.direction.filters import BaseFilter


class PlaceCountFilter(BaseFilter):
    VALUES = [1, 2, 3, 4]

    def __init__(self, selected):
        self._selected = selected
        super(PlaceCountFilter, self).__init__(self._selected)

    @classmethod
    def load(cls, values):
        if len(values) == 0:
            value = None
        else:
            if len(values) != 1:
                raise serializers.ValidationError(
                    'invalid place count filter value: you have sent many values, but you should send only one value'
                )
            try:
                value = int(values[0])
                if value not in cls.VALUES:
                    raise RuntimeError('Not supported value')
            except (ValueError, RuntimeError):
                raise serializers.ValidationError('invalid place count filter value: values should be one of [{}]'.format(
                    ', '.join((str(x) for x in cls.VALUES))
                ))

        return cls(value)

    def bind(self, variants):
        variant_values = []
        for variant in variants:
            value = 0
            places_group = variant.places_group
            if places_group is not None and places_group.max_seats_in_the_same_car > 0:
                value = self.VALUES[bisect.bisect(self.VALUES, places_group.max_seats_in_the_same_car) - 1]

            variant_values.append(value)

        self._variant_values = variant_values

    def list_selectors(self):
        assert self.is_bound
        selected, variant_values = self._selected, self._variant_values

        if selected is None:
            return [True] * len(variant_values)

        return [variant_value >= selected and variant_value != 0 for variant_value in variant_values]

    def dump(self):
        selected, available = self._selected, self._available

        max_passengers = 0
        if available:
            max_passengers = max(available)
        return [
            {
                'value': value,
                'selected': value == selected,
                'available': value <= max_passengers
            }
            for value in self.VALUES
        ]
