# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import bisect

from rest_framework import serializers

from travel.rasp.wizards.wizard_lib.direction.filters import BaseFilter


class PriceFilter(BaseFilter):
    VALUES, BOUNDS = zip(('-1000', 0),
                         ('1000-2000', 1000),
                         ('2000-3000', 2000),
                         ('3000-', 3000))

    def __init__(self, selected):
        self._selected = selected
        super(PriceFilter, self).__init__(self._selected)

    @classmethod
    def load(cls, values):
        selected = set()
        for value in values:
            if value not in cls.VALUES:
                raise serializers.ValidationError('invalid price filter value: it should be one of [{}]'
                                                  .format(', '.join(cls.VALUES)))
            selected.add(value)
        return cls(selected)

    def bind(self, variants):
        variant_values = []
        for v in variants:
            places_group = v.places_group
            if places_group is None:
                variant_values.append(None)
            else:
                variant_values.append(
                    self.VALUES[bisect.bisect(
                        self.BOUNDS, places_group.price.value
                    ) - 1]
                )
        self._variant_values = variant_values

    def list_selectors(self):
        assert self.is_bound
        selected, variant_values = self._selected, self._variant_values

        if not selected:
            return [True] * len(variant_values)

        return [variant_value in selected for variant_value in variant_values]

    def dump(self):
        selected, available = self._selected, self._available
        return [
            {
                'value': range_value,
                'selected': range_value in selected,
                'available': range_value in available
            }
            for range_value in self.VALUES
        ]
