# coding: utf8
from __future__ import absolute_import, division, print_function, unicode_literals

import os
import sys
import logging

from jaeger_client import Config
from travel.library.python.tracing.django import Tracing
from travel.library.python.tracing.gunicorn import build_post_fork


os.environ.setdefault("DJANGO_SETTINGS_MODULE", "local_settings")

sys.path.insert(0, os.path.join(os.path.dirname(__file__), '..'))

from travel.rasp.library.python.common23.server.gunicorn.default import *  # noqa

bind = "unix:/tmp/train-wizard-api.sock"
workers = os.environ.get('TRAIN_WIZARD_API_GUNICORN_WORKERS', 2)
limit_request_line = 8190

log = logging.getLogger(__name__)

jaeger_sampler_type = os.environ.get('JAEGER_SAMPLER_TYPE', 'probabilistic')
jaeger_sampler_parameter = float(os.environ.get('JAEGER_SAMPLER_PARAMETER', 0.001))
jaeger_config = Config(
    config={
        'sampler': {
            'type': jaeger_sampler_type,
            'param': jaeger_sampler_parameter,
        },
        'logging': True,
    },
    service_name='train-wizard-api',
    validate=True,
)


def get_postgres_hosts():
    from django.conf import settings
    from travel.rasp.library.python.common23.data_api.dbaas.client import HostInfo, DbType
    from travel.rasp.library.python.common23.data_api.dbaas.instance import dbaas_client
    from travel.rasp.library.python.common23.data_api.dbaas.host_sorter import host_sorter

    try:
        if settings.DBAAS_TRAIN_WIZARD_API_DB_HOST is None:
            db_hosts = dbaas_client.get_hosts(
                cluster_id=settings.DBAAS_TRAIN_WIZARD_API_DB_CLUSTER_ID,
                db_type=DbType.POSTGRESQL
            )
        else:
            db_hosts = [HostInfo(settings.DBAAS_TRAIN_WIZARD_API_DB_HOST, 'xxx')]
    except Exception:
        # tmp hardcode to fix in prod
        env_type = os.getenv('YENV_TYPE')
        if env_type == 'production':
            log.exception("Can't get pg hosts from dbaas api, use hardcoded hosts")
            db_hosts = [
                HostInfo('sas-4xwhe5p3g2lhvl7s.db.yandex.net', 'sas'),
                HostInfo('vla-o0qyi112wujrppgt.db.yandex.net', 'vla'),
                HostInfo('man-g31pouyrkr47d5sn.db.yandex.net', 'man'),
            ]
            db_hosts = host_sorter.sort(db_hosts)
        elif env_type == 'testing':
            log.exception("Can't get pg hosts from dbaas api, use hardcoded hosts")
            db_hosts = [
                HostInfo('sas-g060r8k32sazdjnx.db.yandex.net', 'sas'),
                HostInfo('vla-tev4b17pb7fffbiq.db.yandex.net', 'vla'),
                HostInfo('man-s1t2m98fxv1m25lp.db.yandex.net', 'man'),
            ]
            db_hosts = host_sorter.sort(db_hosts)
        else:
            raise

    return db_hosts


tracing_post_fork = build_post_fork(jaeger_config, wrapper_cls=Tracing)


def post_fork(server, worker):
    from travel.rasp.wizards.train_wizard_api.lib.storage_store import storage_store

    db_hosts = get_postgres_hosts()

    storage_store.init(db_hosts, alias='slave')
    storage_store.init(db_hosts, alias='master', allow_read_only=False)

    server.log.info('DBAAS hosts: %r', db_hosts)

    tracing_post_fork(server, worker)


_default_when_ready = when_ready  # noqa: ImportStarUsage


def when_ready(server):
    from travel.rasp.wizards.wizard_lib.utils.timestamp_checker import setup_timestamped_recache

    _default_when_ready(server)
    setup_timestamped_recache(code='prepare_all', logger=server.log)
