# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

from contextlib2 import contextmanager

from common.apps.train.models import Facility


class FacilityProvider(object):
    def __init__(self):
        # type: () -> None
        super(FacilityProvider, self).__init__()

        self._code_to_id = {}
        self._id_to_code = {}

    @contextmanager
    def using_precache(self):
        if not self.is_precached():
            self.build_cache()
            try:
                yield
            finally:
                self.clean()
        else:
            yield

    def build_cache(self):
        query = Facility.objects.values_list(
            'id',
            'code',
        )

        self._code_to_id = {}
        self._id_to_code = {}
        for pk, code in query:
            self._code_to_id[code] = pk
            self._id_to_code[pk] = code

    def clean(self):
        self._code_to_id = {}
        self._id_to_code = {}

    def get_id_by(self, code):
        return self._code_to_id.get(code)

    def get_code_by(self, pk):
        return self._id_to_code.get(pk)

    def get_all_codes(self):
        return frozenset(self._code_to_id.keys())

    def get_all_ids(self):
        return frozenset(self._id_to_code.keys())

    def is_precached(self):
        return bool(self._code_to_id)


facility_provider = FacilityProvider()
