# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

from concurrent.futures import Future
from datetime import timedelta, date, datetime, time  # noqa: UnusedImport
from logging import Logger, getLogger  # noqa: UnusedImport

from sqlalchemy import sql
from typing import Tuple, List, Set  # noqa: UnusedImport

import common.utils.railway as railway_utils
from common.models_utils.geo import Point  # noqa: UnusedImport
from travel.library.python.tracing.instrumentation import traced_function
from travel.rasp.wizards.train_wizard_api.lib.express_system_provider import ExpressSystemProvider, express_system_provider  # noqa: UnusedImport
from travel.rasp.wizards.train_wizard_api.lib.facility_provider import FacilityProvider, facility_provider  # noqa: UnusedImport
from travel.rasp.wizards.train_wizard_api.lib.pgaas_price_store.db_models import TrainInfo
from travel.rasp.wizards.train_wizard_api.lib.storage_store import StorageStore, storage_store  # noqa: UnusedImport
from travel.rasp.wizards.train_wizard_api.lib.storage_timed_execute import ExecutionTimeout, execute_with_timeout, get_future


class TrainInfoModel(object):
    def __init__(self, departure_at, number, facilities_ids, electronic_ticket, updated_at):
        # type: (datetime, unicode, List[int], bool, datetime) -> None
        self.departure_at = departure_at
        self.number = number
        self.facilities_ids = facilities_ids
        self.electronic_ticket = electronic_ticket
        self.updated_at = updated_at

    def __repr__(self):
        return "<TrainInfoModel departure_at={} number={} updated_at={} facilities_ids={}/>".format(
            self.departure_at,
            self.number,
            self.updated_at,
            self.facilities_ids,
        )

    def __eq__(self, other):
        if not isinstance(other, self.__class__):
            return False

        return (
            self.departure_at == other.departure_at and
            self.number == other.number and
            self.updated_at == other.updated_at and
            self.facilities_ids == other.facilities_ids
        )


class TrainInfoProvider(object):
    def __init__(self, storage_store, express_system_provider, railway_utils, facility_provider, logger):
        # type: (StorageStore, ExpressSystemProvider, any, FacilityProvider, Logger) -> None
        self._storage_store = storage_store
        self._express_system_provider = express_system_provider
        self._railway_utils = railway_utils
        self._facility_provider = facility_provider
        self._logger = logger

    @traced_function(name='train_wizard_api.lib.pgaas_price_store.tariff_direction_info_provider.TrainInfoProvider.find')
    def find(self, departure_point, arrival_point, departure_date, days=1):
        # type: (Point, Point, date, int) -> Tuple[TrainInfoModel, ...]

        departure_point_key = departure_point.point_key
        arrival_point_key = arrival_point.point_key

        departure_point_express_ids = self._express_system_provider.find_related_express_ids(departure_point_key)
        arrival_point_express_ids = self._express_system_provider.find_related_express_ids(arrival_point_key)

        if not departure_point_express_ids or not arrival_point_express_ids:
            self._logger.warn(
                'Can not find prices by [%s-%s], because can not find express codes for one of points [%s-%s]',
                departure_point_express_ids, arrival_point_express_ids,
                departure_point_key, arrival_point_key,
            )
            result = Future()
            result.set_result(())
            return result, {}

        self._logger.info(
            'Start search: [%s(%s)-%s(%s)-%s-%s]',
            departure_point_key, departure_point_express_ids, arrival_point_key,
            arrival_point_express_ids, departure_date, days
        )

        left_border = datetime.combine(departure_date - timedelta(days=1), time.min)
        right_border = datetime.combine(departure_date + timedelta(days=days), time.max)

        return self._find(departure_point_express_ids, arrival_point_express_ids, left_border, right_border), ()

    def _find(self, departure_point_express_ids, arrival_point_express_ids, left_border, right_border):
        # type: (Set[int], Set[int], datetime, datetime) -> Tuple[TrainInfoModel, ...]
        storage = self._storage_store.get('slave')
        query = sql.select([
            TrainInfo.departure_at,
            TrainInfo.number,
            TrainInfo.facilities_ids,
            TrainInfo.electronic_ticket,
            TrainInfo.updated_at
        ], whereclause=sql.and_(
            TrainInfo.departure_point_express_id.in_(departure_point_express_ids),
            TrainInfo.arrival_point_express_id.in_(arrival_point_express_ids),
            TrainInfo.departure_at >= left_border,
            TrainInfo.departure_at <= right_border,
        ), order_by=[
            TrainInfo.departure_at,
            TrainInfo.number
        ])

        return get_future(storage, query)

    @traced_function(
        name='train_wizard_api.lib.pgaas_price_store.tariff_direction_info_provider.TrainInfoProvider.build_info')
    def build_info(self, rows, context):
        self._logger.error('TrainInfoProvider build_info')
        return tuple(self._create_train_info_model(*r) for r in rows)

    @traced_function(
        name='train_wizard_api.lib.pgaas_price_store.tariff_direction_info_provider.TrainInfoProvider.build_empty_info')
    def build_empty_info(self, context):
        self._logger.error('TrainInfoProvider empty_info')
        return ()

    def _create_train_info_model(self, departure_at, number, facilities_ids, electronic_ticket, updated_at):
        return TrainInfoModel(
            departure_at=departure_at,
            number=number,
            facilities_ids=[
                pk for pk in facilities_ids
                if self._facility_provider.get_code_by(pk)
            ],
            electronic_ticket=electronic_ticket,
            updated_at=updated_at,
        )


train_info_provider = TrainInfoProvider(
    storage_store=storage_store,
    express_system_provider=express_system_provider,
    railway_utils=railway_utils,
    facility_provider=facility_provider,
    logger=getLogger(__name__)
)
