# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

from logging import getLogger
from typing import List  # noqa: UnusedImport

import psycopg2.extras
from django.conf import settings
from sqlalchemy import create_engine
from sqlalchemy.engine import Engine  # noqa: UnusedImport
from sqlalchemy.orm import sessionmaker, Session  # noqa: UnusedImport

from travel.rasp.library.python.common23.data_api.dbaas.client import HostInfo  # noqa: UnusedImport


def make_logging_connection_factory(logger):
    def connection_factory(*args, **kwargs):
        connection = psycopg2.extras.LoggingConnection(*args, **kwargs)
        connection.initialize(logger)
        return connection

    return connection_factory


class Storage(object):
    def __init__(self, connection_string, connect_args):
        self._engine = create_engine(
            connection_string,
            connect_args=connect_args,
            pool_recycle=settings.DBAAS_TRAIN_WIZARD_API_POOL_RECYCLE,
            pool_size=2
        )
        self._session_factory = sessionmaker(bind=self._engine)

    def get_session(self):
        # type: () -> Session
        return self._session_factory()

    def get_engine(self):
        # type: () -> Engine
        return self._engine


class StorageStore(object):
    def __init__(self):
        self._alias_to_storage = {}

    def init(self, hosts, alias, allow_read_only=True, logging=False):
        # type: (List[HostInfo], unicode, bool, bool) -> None

        if alias in self._alias_to_storage:
            raise RuntimeError("Can not rewrite db, please use another alias")

        connect_args = {
            "sslmode": settings.DBAAS_TRAIN_WIZARD_API_SSL_MODE,
            "sslrootcert": settings.INTERNAL_ROOT_CERT,
            "target_session_attrs": "any" if allow_read_only else "read-write",
        }
        if logging:
            connect_args["connection_factory"] = make_logging_connection_factory(getLogger(__name__))

        connection_string = "postgresql+psycopg2://{user}:{password}@{host}:{port}/{dbname}".format(
            user=settings.DBAAS_TRAIN_WIZARD_API_DB_USER,
            password=settings.DBAAS_TRAIN_WIZARD_API_DB_PASSWORD,
            host=",".join(h.host for h in hosts),
            port=settings.DBAAS_TRAIN_WIZARD_API_DB_PORT,
            dbname=settings.DBAAS_TRAIN_WIZARD_API_DB_NAME,
        )

        self._alias_to_storage[alias] = storage = Storage(connection_string, connect_args)
        return storage

    def get(self, alias):
        # type: (unicode) -> Storage
        if alias not in self._alias_to_storage:
            raise RuntimeError("Can not find storage by alias:[{}]. Please Init database before use it".format(alias))
        return self._alias_to_storage[alias]


storage_store = StorageStore()
