# coding: utf-8
from __future__ import absolute_import, division, print_function, unicode_literals

import logging
from contextlib import closing

from concurrent.futures import ThreadPoolExecutor, TimeoutError

from common.utils.yasmutil import MeasurableDecorator, Metric, YasmMetricSender
from opentracing import global_tracer
from travel.library.python.tracing.instrumentation import child_span, traced_function

logger = logging.getLogger(__name__)

YASM_PREFIX = 'dbaas_storage'


class measurable(MeasurableDecorator):
    prefix = YASM_PREFIX


@measurable('timed_execute')
def _execute_session_query(storage, query, active_span):
    if active_span is not None:
        with global_tracer().scope_manager.activate(active_span, True) as scope:
            with closing(storage.get_session()) as session:
                return session.execute(query)
    with closing(storage.get_session()) as session:
        return session.execute(query)


class ExecutionTimeout(Exception):
    pass


class TimedExecutor(object):
    def __init__(self):
        self._executor = ThreadPoolExecutor(max_workers=3)

    def execute_with_timeout(self, storage, query, timeout):
        active_span = getattr(global_tracer().scope_manager.active, 'span', None)
        future = self._executor.submit(_execute_session_query, storage, query, active_span)
        try:
            return future.result(timeout)
        except TimeoutError:
            try:
                YasmMetricSender(prefix=YASM_PREFIX).send_one(Metric('timeouts_cnt', 1, 'ammm'))
            except Exception:
                logger.exception('cannot send yasm metric')
            raise ExecutionTimeout

    @traced_function(name='TimedExecutor.execute_all_with_timeout')
    def execute_all_with_timeout(self, parameters, timeout):
        futures = [
            (provider, provider.find(*provider_params))
            for provider, provider_params in parameters
        ]

        return [
            wait_for_future_and_build_info(provider, future, context, timeout)
            for provider, (future, context) in futures
        ]

    def get_future(self, storage, query):
        active_span = getattr(global_tracer().scope_manager.active, 'span', None)
        return self._executor.submit(_execute_session_query, storage, query, active_span)

    @staticmethod
    @traced_function(name='TimedExecutor.wait_for_future_and_build_info')
    def wait_for_future_and_build_info(provider, future, context, timeout):
        try:
            rows = future.result(timeout)
            return provider.build_info(rows, context)
        except TimeoutError:
            logger.error('{}.find timeout'.format(type(provider).__name__))
            try:
                YasmMetricSender(prefix=YASM_PREFIX).send_one(Metric('timeouts_cnt', 1, 'ammm'))
            except Exception:
                logger.exception('cannot send yasm metric')
            return provider.build_empty_info(context)

    def close(self):
        self._executor.shutdown()


default_executor = TimedExecutor()
execute_with_timeout = default_executor.execute_with_timeout
get_future = default_executor.get_future
wait_for_future_and_build_info = default_executor.wait_for_future_and_build_info
execute_all_with_timeout = default_executor.execute_all_with_timeout
