# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import itertools
from contextlib2 import contextmanager

from common.models.geo import StationCode, StationMajority
from common.models.transport import TransportType


class TrainCityProvider(object):
    def __init__(self):
        self._cache = None

    @contextmanager
    def using_precache(self):
        if not self.is_precached():
            self.build_cache()
            try:
                yield
            finally:
                self.clean()
        else:
            yield

    def is_precached(self):
        return self._cache is not None

    def clean(self):
        self._cache = None

    def build_cache(self):
        city_id_to_station_id = {}

        fake_query = StationCode.objects.filter(
            station__t_type__id=TransportType.TRAIN_ID,
            station__settlement_id__isnull=False,
            system__code='express',
            station__majority_id=StationMajority.EXPRESS_FAKE_ID
        ).order_by(
            'station__majority_id'
        ).values_list(
            'station_id',
            'station__settlement_id',
            'station__majority_id'
        )

        query = StationCode.objects.filter(
            station__t_type__id=TransportType.TRAIN_ID,
            station__settlement_id__isnull=False,
            system__code='express',
            station__hidden=False
        ).order_by(
            'station__majority_id'
        ).values_list(
            'station_id',
            'station__settlement_id',
            'station__majority_id'
        )

        for station_id, settlement_id, majority in itertools.chain(fake_query, query):
            if settlement_id not in city_id_to_station_id:
                city_id_to_station_id[settlement_id] = station_id

        self._cache = city_id_to_station_id

    def find_train_station_id(self, city_id):
        assert self.is_precached(), 'Provider is not initialized, please initialize It before use'
        return self._cache.get(city_id)


train_city_provider = TrainCityProvider()
