# coding: utf8
from __future__ import absolute_import, division, print_function, unicode_literals

import urlparse
from urllib import quote_plus

from django.conf import settings
from django.utils.encoding import force_bytes
from django.utils.http import urlencode


TRAVEL_BASE_TRAIN_PREFIX = 'trains'


class RaspSearchFactory(object):
    def __init__(self, host_by_tld):
        self._host_by_tld = host_by_tld

    def format_url(self, from_point, to_point, when, tld, transport_type, empty_variants, extra_params=()):
        tld = tld if tld in self._host_by_tld else 'ru'
        host = self._host_by_tld[tld]

        query_string = urlencode(
            (
                ('fromName', from_point.L_title()),
                ('fromId', from_point.point_key),
                ('toName', to_point.L_title()),
                ('toId', to_point.point_key),
                ('when', when if isinstance(when, basestring) else when.isoformat()),
            ) + extra_params
        )

        return urlparse.SplitResult(
            scheme='https',
            netloc=host,
            path='/search/{}/'.format(transport_type),
            query=query_string,
            fragment=None
        ).geturl()


class TrainSearchFactory(object):
    def __init__(self, host_by_tld, path_prefix=''):
        self._host_by_tld = host_by_tld
        self._path_prefix = path_prefix

    def format_url(self, from_point, to_point, when, tld, transport_type, empty_variants, extra_params=()):
        assert transport_type == 'train'
        tld = tld if tld in self._host_by_tld else 'ru'
        host = self._host_by_tld[tld]

        if empty_variants:
            path = '{}/'.format(self._path_prefix)
        else:
            path = '{}/{}--{}/'.format(
                self._path_prefix, from_point.slug, to_point.slug,
            )
            when_param = ('when', when.isoformat()),
            extra_params = when_param + extra_params

        query_string = urlencode(extra_params) if extra_params else None

        return urlparse.SplitResult(
            scheme='https',
            netloc=host,
            path=path,
            query=query_string,
            fragment=None
        ).geturl()


class FactorySelector(object):
    def __init__(self, rasp_desktop_url_factory, rasp_mobile_url_factory, travel_url_factory):
        self._rasp_desktop_url_factory = rasp_desktop_url_factory
        self._rasp_mobile_url_factory = rasp_mobile_url_factory
        self._travel_url_factory = travel_url_factory

    def select_factory(self, is_mobile, transport_type):
        if is_mobile:
            rasp_factory = self._rasp_mobile_url_factory
        else:
            rasp_factory = self._rasp_desktop_url_factory

        if transport_type == 'train':
            return self._travel_url_factory

        return rasp_factory


class HostProvider(object):
    def __init__(self, host_by_tld, path_prefix=''):
        self._host_by_tld = host_by_tld
        self._path_prefix = path_prefix

    def get_host(self, tld):
        tld = tld if tld in self._host_by_tld else 'ru'
        if self._path_prefix:
            return '{}/{}/'.format(self._host_by_tld[tld], self._path_prefix)
        return self._host_by_tld[tld]


class ProxyHostProvider(object):
    def __init__(self, factory_selector):
        self._factory_selector = factory_selector

    def get_host(self, is_mobile, transport_type, tld):
        return self._factory_selector.select_factory(
            is_mobile,
            transport_type,
        ).get_host(
            tld=tld
        )


class ProxySearchUrlFactory(object):
    def __init__(self, factory_selector):
        self._factory_selector = factory_selector

    def format_url(self, is_mobile, from_point, to_point, when, tld, transport_type, empty_variants, extra_params=()):
        return self._factory_selector.select_factory(
            is_mobile,
            transport_type,
        ).format_url(
            from_point=from_point,
            to_point=to_point,
            when=when,
            tld=tld,
            transport_type=transport_type,
            empty_variants=empty_variants,
            extra_params=extra_params,
        )


class OrderUrlContext(object):
    def __init__(self, prefix_url):
        self._prefix_url = prefix_url

    def add_segment_info(self, number, departure_local_dt, provider):
        new_url = b'{prefix}&number={number}&time={deparure_dt:%H:%M}'.format(
            prefix=self._prefix_url,
            number=quote_plus(force_bytes(number)),
            deparure_dt=departure_local_dt
        )
        if provider:
            new_url += b'&provider={provider}'.format(provider=provider)
        return OrderUrlContext(new_url)

    def add_coach_info(self, coach_type):
        return OrderUrlContext(b'{prefix}&coachType={coach_type}'.format(
            prefix=self._prefix_url,
            coach_type=coach_type
        ))

    def format_url(self):
        return self._prefix_url


class OrderFactory(object):
    def __init__(self, host_by_tld, path_prefix=''):
        self._host_by_tld = host_by_tld
        self._path_prefix = path_prefix

    def build_context(self, from_point, to_point, when, tld, transport_type, main_reqid=None):
        tld = tld if tld in self._host_by_tld else 'ru'
        host = self._host_by_tld[tld]

        parameters = [
            ('fromName', from_point.L_title()),
            ('fromId', from_point.point_key),
            ('toName', to_point.L_title()),
            ('toId', to_point.point_key),
            ('when', when.isoformat()),
            ('transportType', transport_type),
        ]
        if main_reqid is not None:
            parameters.append(('wizardReqId', main_reqid))
        query_string = urlencode(tuple(parameters))

        prefix_url = urlparse.SplitResult(
            scheme='https',
            netloc=host,
            path='{}/order/'.format(self._path_prefix),
            query=query_string,
            fragment=None
        ).geturl()

        return OrderUrlContext(prefix_url)


class ProxyOrderFactory(object):
    def __init__(self, factory_selector):
        self._factory_selector = factory_selector

    def build_context(self, is_mobile, from_point, to_point, when, tld, transport_type, main_reqid=None):
        return self._factory_selector.select_factory(
            is_mobile,
            transport_type,
        ).build_context(from_point, to_point, when, tld, transport_type, main_reqid)


host_provider = ProxyHostProvider(
    factory_selector=FactorySelector(
        rasp_desktop_url_factory=HostProvider(
            settings.MORDA_HOST_BY_TLD
        ),
        rasp_mobile_url_factory=HostProvider(
            settings.TOUCH_HOST_BY_TLD
        ),
        travel_url_factory=HostProvider(
            settings.TRAVEL_HOST_BY_TLD, TRAVEL_BASE_TRAIN_PREFIX,
        ),
    )
)

search_url_factory = ProxySearchUrlFactory(
    factory_selector=FactorySelector(
        rasp_desktop_url_factory=RaspSearchFactory(
            settings.MORDA_HOST_BY_TLD
        ),
        rasp_mobile_url_factory=RaspSearchFactory(
            settings.TOUCH_HOST_BY_TLD
        ),
        travel_url_factory=TrainSearchFactory(
            settings.TRAVEL_HOST_BY_TLD, TRAVEL_BASE_TRAIN_PREFIX,
        ),
    )
)

order_url_factory = ProxyOrderFactory(
    factory_selector=FactorySelector(
        rasp_desktop_url_factory=OrderFactory(
            settings.MORDA_HOST_BY_TLD
        ),
        rasp_mobile_url_factory=OrderFactory(
            settings.TOUCH_HOST_BY_TLD
        ),
        travel_url_factory=OrderFactory(
            settings.TRAVEL_HOST_BY_TLD, TRAVEL_BASE_TRAIN_PREFIX
        ),
    )
)
