# coding: utf-8
from __future__ import absolute_import, division, print_function, unicode_literals

from collections import namedtuple

from rest_framework import serializers

from travel.library.python.tracing.instrumentation import traced_function
from travel.rasp.wizards.wizard_lib.serialization.date import parse_date
from travel.rasp.wizards.wizard_lib.serialization.experiment_flags import parse_experiment_flags
from travel.rasp.wizards.wizard_lib.serialization.language import parse_language
from travel.rasp.wizards.wizard_lib.serialization.point import dump_point, parse_point

DEFAULT_DAYS = 5


def _parse_days(value):
    if value:
        try:
            days = int(value)
        except ValueError:
            pass
        else:
            if days > 0:
                return days

        raise serializers.ValidationError('invalid days value: it should be positive integer')

    return DEFAULT_DAYS


DirectionPlacesQuery = namedtuple('DirectionPlacesQuery', ('departure_point', 'arrival_point', 'min_departure_date',
                                                           'days', 'language', 'tld', 'experiment_flags'))


@traced_function
def load_query(query_params):
    departure_point = parse_point(query_params, 'departure_point_key', 'departure_settlement_geoid')
    return DirectionPlacesQuery(
        departure_point=departure_point,
        arrival_point=parse_point(query_params, 'arrival_point_key', 'arrival_settlement_geoid'),
        min_departure_date=parse_date(query_params.get('min_departure_date'), departure_point.pytz, today_default=True),
        days=_parse_days(query_params.get('days')),
        language=parse_language(query_params.get('language')),
        experiment_flags=parse_experiment_flags(query_params.get('exp_flags')),
        tld=query_params.get('tld')
    )


@traced_function
def dump_query(query):
    return {
        'query': {
            'departure_point': dump_point(query.departure_point),
            'arrival_point': dump_point(query.arrival_point),
            'min_departure_date': query.min_departure_date.strftime('%Y-%m-%d'),
            'language': query.language,
            'days': query.days
        },
    }
