# coding: utf-8
from __future__ import absolute_import, division, print_function, unicode_literals

import itertools
from urllib import quote_plus

from django.utils.datastructures import MultiValueDict
from django.utils.encoding import force_bytes
from django.utils.translation import ugettext
from rest_framework import serializers

from common.models.geo import Country, Region
from common.utils.date import timedelta2minutes
from travel.rasp.wizards.train_wizard_api.direction.filters import Filters
from travel.rasp.wizards.train_wizard_api.direction.sorting import DEFAULT_SORTING
from travel.rasp.wizards.train_wizard_api.lib.url_factories import order_url_factory, search_url_factory
from travel.rasp.wizards.train_wizard_api.serialization.route import find_route_with_train_stations
from travel.rasp.wizards.wizard_lib.serialization.date import parse_date
from travel.rasp.wizards.wizard_lib.serialization.direction import DirectionQuery
from travel.rasp.wizards.wizard_lib.serialization.experiment_flags import parse_experiment_flags
from travel.rasp.wizards.wizard_lib.serialization.language import parse_language
from travel.rasp.wizards.wizard_lib.serialization.point import parse_point
from travel.rasp.wizards.wizard_lib.serialization.price import dump_price
from travel.rasp.wizards.wizard_lib.serialization.segments import dump_duration, get_minimum_duration


SEGMENTS_LIMIT = 5


def _dump_datetime(local_dt):
    return local_dt.strftime('%Y-%m-%d %H:%M:%S %z')


def _dump_train_brand(train_brand):
    return train_brand.L_title() if train_brand else None


def _dump_minimum_price(segment_prices):
    segment_prices = filter(None, segment_prices)
    if segment_prices:
        return dump_price(min(segment_prices))


def _format_segment_url_query(search_url_query, segment):
    return b'{search_url_query}&transportType=train&number={number}&time={deparure_dt:%H:%M}'.format(
        search_url_query=search_url_query,
        number=quote_plus(force_bytes(segment.train_number)),
        deparure_dt=segment.departure_local_dt
    )


def _dump_places_group(places_group, mobile_order_url_context, desktop_order_url_context):
    mobile_order_url_context = mobile_order_url_context.add_coach_info(coach_type=places_group.coach_type)
    desktop_order_url_context = desktop_order_url_context.add_coach_info(coach_type=places_group.coach_type)

    return {
        'coach_type': places_group.coach_type,
        'count': places_group.count,
        'order_touch_url': mobile_order_url_context.format_url(),
        'order_url': desktop_order_url_context.format_url(),
        'price': dump_price(places_group.price),
    }


def _dump_segment(segment, mobile_order_url_context, desktop_order_url_context, language):
    mobile_order_url_context = mobile_order_url_context.add_segment_info(
        segment.train_number, segment.departure_local_dt, segment.provider
    )
    desktop_order_url_context = desktop_order_url_context.add_segment_info(
        segment.train_number, segment.departure_local_dt, segment.provider
    )

    return {  # TODO: поля aeroexpress, express
        'arrival': _dump_datetime(segment.arrival_local_dt),
        'arrival_station': segment.arrival_station.L_popular_title(),
        'brand': _dump_train_brand(segment.train_brand),
        'departure': _dump_datetime(segment.departure_local_dt),
        'departure_station': segment.departure_station.L_popular_title(),
        'duration': timedelta2minutes(segment.arrival_local_dt - segment.departure_local_dt),
        'from_station': ugettext('from_station') % {
            'station_title': segment.departure_station.L_popular_title(case='genitive')
        },
        'number': segment.train_number,
        'order_touch_url': mobile_order_url_context.format_url(),
        'order_url': desktop_order_url_context.format_url(),
        'places': None if segment.places is None else [
            _dump_places_group(places_group, mobile_order_url_context, desktop_order_url_context)
            for places_group in segment.places
        ],
        'title': getattr(segment.train_title, language),
    }


def dump_segments(segments, departure_point, arrival_point, when, language, tld, experiment_flags):
    total_segments = len(segments)
    segments = segments[:SEGMENTS_LIMIT]
    segment_prices = tuple(
        None if not segment.places else min(place.price for place in segment.places)
        for segment in segments
    )

    mobile_order_url_context = order_url_factory.build_context(
        is_mobile=True,
        from_point=departure_point,
        to_point=arrival_point,
        when=when,
        tld=tld,
        transport_type='train',
    )

    desktop_order_url_context = order_url_factory.build_context(
        is_mobile=False,
        from_point=departure_point,
        to_point=arrival_point,
        when=when,
        tld=tld,
        transport_type='train',
    )

    minimum_duration = get_minimum_duration(segments) if segments else None

    return {
        'minimum_duration': dump_duration(minimum_duration),
        'minimum_price': _dump_minimum_price(segment_prices),
        'segments': [
            dict(_dump_segment(
                segment=segment,
                mobile_order_url_context=mobile_order_url_context,
                desktop_order_url_context=desktop_order_url_context,
                language=language
            ), **{'price': dump_price(segment_price)})
            for segment, segment_price in itertools.izip(segments, segment_prices)
        ],
        'total': total_segments,
        'touch_url': search_url_factory.format_url(
            is_mobile=True,
            from_point=departure_point,
            to_point=arrival_point,
            when=when,
            tld=tld,
            empty_variants=not segments,
            transport_type='train',
        ),
        'transport': 'train',
        'url': search_url_factory.format_url(
            is_mobile=False,
            from_point=departure_point,
            to_point=arrival_point,
            when=when,
            tld=tld,
            empty_variants=not segments,
            transport_type='train',
        ),
    }


def load_query(query_params):
    experiment_flags = parse_experiment_flags(query_params.get('exp_flags'))
    departure_point, arrival_point, _, _ = find_route_with_train_stations(query_params, experiment_flags)

    if departure_point is None or arrival_point is None:
        departure_point = parse_point(
            query_params=query_params,
            point_key_name='departure_point_key',
            settlement_geoid_name='departure_settlement_geoid',
        )
        arrival_point = parse_point(
            query_params=query_params,
            point_key_name='arrival_point_key',
            settlement_geoid_name='arrival_settlement_geoid',
        )

    if not (departure_point.country_id == arrival_point.country_id == Country.RUSSIA_ID):
        raise serializers.ValidationError('direction points should be within Russia')

    if (
        departure_point.region_id == Region.KALININGRAD_REGION_ID
    ) ^ (
        arrival_point.region_id == Region.KALININGRAD_REGION_ID
    ):
        raise serializers.ValidationError('crossing of Kaliningrad region border is not alowed')

    main_reqid = query_params.get('main_reqid') or None

    return DirectionQuery(
        departure_point=departure_point,
        arrival_point=arrival_point,
        original_departure_point=None,
        original_arrival_point=None,
        departure_date=parse_date(
            value=query_params.get('date'),
            local_tz=departure_point.pytz,
            ignore_past=False,
        ),
        filters=Filters.load(MultiValueDict({'coach_type': query_params.getlist('coach_type')})),
        language=parse_language(query_params.get('lang')),
        experiment_flags=experiment_flags,
        sorting=DEFAULT_SORTING,
        tld=query_params.get('tld'),
        limit=None,
        main_reqid=main_reqid,
    )
