# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

from collections import namedtuple
from datetime import datetime
from itertools import izip

from marshmallow import Schema, fields, validates_schema, validate, post_load

from common.models.geo import Station
from common.serialization.common_schemas import MultiValueDictSchemaMixin
from common.serialization.fields import PointField
from travel.rasp.wizards.train_wizard_api.lib.url_factories import order_url_factory
from travel.rasp.wizards.wizard_lib.serialization.place import dump_tarif_info_places


TariffsByDirectionQuery = namedtuple('TariffsByDirectionQuery', [
    'directions', 'departure_date_from', 'departure_date_to', 'tld'
])


class TariffsByDirectionQuerySchema(Schema, MultiValueDictSchemaMixin):
    departure_points = fields.List(PointField(), required=True)
    arrival_points = fields.List(PointField(), required=True)
    departure_date_from = fields.DateTime(required=True)
    departure_date_to = fields.DateTime(required=True)
    tld = fields.String(required=True)

    @validates_schema
    def validate_direction_points(self, data):
        if len(data.get('departure_points', [])) != len(data.get('arrival_points', [])):
            raise validate.ValidationError('departure_points and arrival_points should be same length')
        return data

    @post_load
    def make_query(self, data):
        directions = [
            (departure, arrival) for departure, arrival
            in izip(data['departure_points'], data['arrival_points'])
        ]
        return TariffsByDirectionQuery(
            directions=directions,
            departure_date_from=data['departure_date_from'],
            departure_date_to=data['departure_date_to'],
            tld=data['tld']
        )


def dump_tariff_info(tariff_info, tld):
    departure_dt = datetime.strptime(tariff_info.departure_dt[:-6], '%Y-%m-%dT%H:%M:%S')
    desktop_order_url_context = order_url_factory.build_context(
        is_mobile=False,
        from_point=Station.objects.get(pk=tariff_info.departure_station_id),
        to_point=Station.objects.get(pk=tariff_info.arrival_station_id),
        when=departure_dt.date(),
        tld=tld,
        transport_type='train',
    )

    return {
        'arrival_dt': tariff_info.arrival_dt,
        'arrival_station_id': tariff_info.arrival_station_id,
        'coach_owners': tariff_info.coach_owners,
        'departure_dt': tariff_info.departure_dt,
        'departure_station_id': tariff_info.departure_station_id,
        'display_number': tariff_info.display_number,
        'electronic_ticket': tariff_info.electronic_ticket,
        'has_dynamic_pricing': tariff_info.has_dynamic_pricing,
        'is_suburban': tariff_info.is_suburban,
        'number': tariff_info.number,
        'order_url': desktop_order_url_context.add_segment_info(
            tariff_info.number, departure_dt, tariff_info.provider).format_url(),
        'places': dump_tarif_info_places(tariff_info),
        'title_dict': tariff_info.title_dict,
        'two_storey': tariff_info.two_storey,
        'first_country_code': tariff_info.first_country_code,
        'last_country_code': tariff_info.last_country_code,
        'provider': tariff_info.provider,
        'raw_train_name': tariff_info.raw_train_name,
    }
