# coding: utf8
from __future__ import absolute_import, division, print_function, unicode_literals

from logging import getLogger

from common.models.geo import Settlement
from travel.rasp.wizards.train_wizard_api.direction.schedule_cache import schedule_cache
from travel.rasp.wizards.train_wizard_api.lib.optimal_direction_provider import optimal_direction_provider
from travel.rasp.wizards.wizard_lib.experiment_flags import ExperimentFlag

log = getLogger(__name__)


OPTIMAL_DIRECTION_RADIUSES = (
    (ExperimentFlag.EXPERIMENTAL_SEARCH_20KM, 20),
    (ExperimentFlag.EXPERIMENTAL_SEARCH_50KM, 50),
    (ExperimentFlag.EXPERIMENTAL_SEARCH_70KM, 70),
    (ExperimentFlag.EXPERIMENTAL_SEARCH, 100),
)

REGION2SETTLEMENT = {977: 146, 10933: 18, 10231: 11319, 11398: 78, 11282: 64, 11443: 74, 11010: 28}


def find_route_with_train_stations(query_params, experiment_flags):
    experiment_flag, radius = next(
        (
            (experiment_flag, radius) for experiment_flag, radius in OPTIMAL_DIRECTION_RADIUSES
            if experiment_flag in experiment_flags
        ),
        (None, None)
    )

    if radius is None:
        return None, None, None, None

    raw_departure_geo_id = query_params.get('departure_settlement_geoid')
    raw_arrival_geo_id = query_params.get('arrival_settlement_geoid')
    try:
        departure_geo_id = int(raw_departure_geo_id)
        arrival_geo_id = int(raw_arrival_geo_id)
        if departure_geo_id in REGION2SETTLEMENT:
            departure_geo_id = REGION2SETTLEMENT[departure_geo_id]
        if arrival_geo_id in REGION2SETTLEMENT:
            arrival_geo_id = REGION2SETTLEMENT[arrival_geo_id]
    except (ValueError, TypeError):
        return None, None, None, None
    departure_point, arrival_point = None, None
    try:
        departure_point = Settlement.hidden_manager.get(_geo_id=departure_geo_id)
        arrival_point = Settlement.hidden_manager.get(_geo_id=arrival_geo_id)
    except (ValueError, Settlement.DoesNotExist):
        pass

    if departure_point and arrival_point:
        if schedule_cache.has_direction(departure_point.id, arrival_point.id):
            # Если есть хоть какие-то прямые маршруты, то ищем по ним
            return None, None, None, None

    nearest_departure_point, nearest_arrival_point = optimal_direction_provider.find_optimal_direction(
        departure_geo_id, arrival_geo_id, radius
    )

    original_departure_point, original_arrival_point = None, None
    try:
        if departure_point is None:
            departure_point = Settlement.objects.get(_geo_id=departure_geo_id)
    except (ValueError, Settlement.DoesNotExist):
        pass
    try:
        if arrival_point is None:
            arrival_point = Settlement.objects.get(_geo_id=arrival_geo_id)
    except (ValueError, Settlement.DoesNotExist):
        pass

    if (nearest_departure_point, nearest_arrival_point) != (None, None):
        log.info(
            "Experimental search with %s: "
            "departure_geo_id: %d arrival_geo_id: %d "
            "from_point: %d to_point: %d",
            experiment_flag.value,
            departure_geo_id,
            arrival_geo_id,
            nearest_departure_point.id,
            nearest_arrival_point.id
        )
        if departure_point != nearest_departure_point:
            original_departure_point = departure_point
        if arrival_point != nearest_arrival_point:
            original_arrival_point = arrival_point
    return nearest_departure_point, nearest_arrival_point, original_departure_point, original_arrival_point
