# coding=utf-8
from __future__ import absolute_import, division, print_function, unicode_literals

import os

from common.settings import *  # noqa: ImportStarUsed
from common.settings.configuration import BaseConfigurator, Configuration
from travel.rasp.library.python.common23.logging.qloud import context_extractor
from travel.rasp.wizards.train_wizard_api.settings.dbaas import *  # noqa
from travel.rasp.wizards.wizard_lib.settings.hosts import (
    TouchHostConfigurator, MordaHostConfigurator, TrainsHostConfigurator, TravelHostConfigurator
)
from travel.rasp.wizards.wizard_lib.settings.locale import LocaleConfigurator


# настройки django

ADMINS = ()

ROOT_URLCONF = 'travel.rasp.wizards.train_wizard_api.urls'

MIDDLEWARE_CLASSES = (
    'django_opentracing.OpenTracingMiddleware',
    'travel.library.python.tracing.django.TracingMiddleware',
    'common.middleware.extract_context.ExtractRequestMiddleware',
    'common.middleware.ping.PingResponder',
)

INSTALLED_APPS = COMMON_INSTALLED_APPS + COMMON_CONTRIB_APPS + [  # noqa: ImportStarUsage
    'django_opentracing',
    'django.contrib.staticfiles',
    'raven.contrib.django.raven_compat',
    'rest_framework',
    'travel.rasp.wizards.train_wizard_api'
]

BASE_DIR = os.path.realpath(os.path.dirname(os.path.dirname(__file__)))

TEMPLATES = [{
    'BACKEND': 'django.template.backends.jinja2.Jinja2',
    'DIRS': [BASE_DIR],  # noqa: ImportStarUsage
    'OPTIONS': {
        'environment': 'travel.rasp.wizards.wizard_lib.utils.jinja2.get_environment',
        'extensions': ['jinja2.ext.i18n'],
        'lstrip_blocks': True,
        'trim_blocks': True,
    }
}]

# настройки third-party модулей

PYLOGCTX_REQUEST_EXTRACTOR = context_extractor

REST_FRAMEWORK = {
    'DEFAULT_RENDERER_CLASSES': ('travel.rasp.wizards.wizard_lib.utils.json_renderer.Renderer',)
}

OPENTRACING_TRACE_ALL = True
OPENTRACING_TRACED_ATTRIBUTES = ['path', 'method']

# настройки common

DISABLE_GROUP_CACHES = True

PRECACHE_ENABLE_FALLBACK_BY_DEFAULT = False

PRECACHE_MANAGERS = [
    'common.models.currency:Currency.objects',
    'common.models.geo:CodeSystem.objects',
    'common.models.geo:Country.objects',
    'common.models.geo:Region.objects',
    'common.models.geo:Settlement.objects',
    'common.models.geo:Station.code_manager',
    'common.models.geo:Station.objects',
    'common.models.geo:StationMajority.objects',
    'common.models.schedule:DeLuxeTrain.objects',
    'common.models.transport:TransportType.objects',
    'common.importinfo.models:Express2Country',
    'travel.rasp.wizards.train_wizard_api.direction.schedule_cache:schedule_cache',
    'travel.rasp.wizards.train_wizard_api.lib.train_city_provider:train_city_provider',
    'travel.rasp.wizards.train_wizard_api.lib.facility_provider:facility_provider',
    'travel.rasp.wizards.train_wizard_api.lib.express_system_provider:express_system_provider',
    'travel.rasp.wizards.wizard_lib.region_capital_provider:region_capital_provider',
    'travel.rasp.wizards.train_wizard_api.lib.optimal_direction_provider:optimal_direction_provider',
]

# настройки приложения

DEV_GENERATE_PRICES = os.environ.get('DEV_GENERATE_PRICES') == 'true'

# настройки, зависящие от окружения

LOG_PATH = '/var/log/app'


class LoggingConfigurator(BaseConfigurator):
    def _configure_logging(self, settings, level='INFO'):
        log_format = '%(levelname)s %(asctime)s %(process)d %(name)s %(message)s'
        log_path = settings['LOG_PATH']

        settings['LOGGING'] = {
            'version': 1,
            'disable_existing_loggers': False,
            'filters': {
                'require_debug_true': {
                    '()': 'django.utils.log.RequireDebugTrue'
                },
                'context_filter': {
                    '()': 'travel.rasp.library.python.common23.logging.AddContextFilter'
                },
                'context_as_dict_filter': {
                    '()': 'travel.rasp.library.python.common23.logging.AddContextFilter',
                    'as_dict': True
                },
            },
            'formatters': {
                'default': {
                    '()': 'ylog.ExceptionFormatter',
                    'format': log_format,
                },
                'verbose': {
                    '()': 'ylog.ExceptionFormatter',
                    'format': log_format,
                    'full': True,
                    'show_locals': True,
                },
                'qloud_application': {
                    '()': 'travel.rasp.library.python.common23.logging.qloud.QloudJsonFormatter',
                    'tag': 'application'
                },
                'qloud_errors': {
                    '()': 'travel.rasp.library.python.common23.logging.qloud.QloudExceptionFormatter',
                    'add_request_info': True,
                    'tag': 'errors'
                },
                'json': {
                    'format': '%(message)s',
                    '()': 'pythonjsonlogger.jsonlogger.JsonFormatter'
                }
            },
            'handlers': {
                'cout': {
                    'class': 'logging.StreamHandler',
                    'filters': ['require_debug_true'],
                    'formatter': 'default',
                },
                'application': {
                    'class': 'travel.rasp.library.python.common23.logging.WatchedFileHandler',
                    'filename': os.path.join(log_path, 'app.log'),
                    'formatter': 'default',
                },
                'db': {
                    'class': 'travel.rasp.library.python.common23.logging.WatchedFileHandler',
                    'filename': os.path.join(log_path, 'db.log'),
                    'formatter': 'default',
                },
                'traceback': {
                    'class': 'travel.rasp.library.python.common23.logging.WatchedFileHandler',
                    'filename': os.path.join(log_path, 'traceback.log'),
                    'formatter': 'verbose',
                },
                'qloud_application': {
                    'class': 'logging.StreamHandler',
                    'stream': 'ext://sys.stdout',
                    'filters': ['context_as_dict_filter'],
                    'formatter': 'qloud_application'
                },
                'qloud_errors': {
                    'class': 'logging.StreamHandler',
                    'stream': 'ext://sys.stderr',
                    'filters': ['context_as_dict_filter'],
                    'formatter': 'qloud_errors'
                },
                'yt.direction': {
                    'class': 'travel.rasp.library.python.common23.logging.WatchedFileHandler',
                    'filename': os.path.join(log_path, 'yt', 'direction.log'),
                    'formatter': 'json',
                    'level': 'INFO',
                },
            },
            'loggers': {
                '': {
                    'handlers': ['application', 'qloud_application'],
                    'level': level,
                },
                'common.db': {
                    'handlers': ['db'],
                    'level': level,
                    'propagate': False,
                },
                'django': {
                    'handlers': ['traceback', 'qloud_errors'],
                    'level': 'ERROR',
                    'propagate': False,
                },
                'django.db': {
                    'handlers': ['db'],
                    'level': 'DEBUG',
                    'propagate': False,
                },
                'kikimr': {
                    'handlers': ['traceback', 'qloud_errors'],
                    'level': 'ERROR',
                    'propagate': False,
                },
                'train_wizard_api': {
                    'handlers': ['cout'],
                    'level': level
                },
                'wizard_lib': {
                    'handlers': ['cout'],
                    'level': level
                },
                'yt.direction': {
                    'handlers': ['yt.direction'],
                    'level': 'INFO',
                    'propagate': False,
                },
            }
        }

    def apply_development(self, settings):
        settings['LOG_PATH'] = os.path.join(settings['PROJECT_PATH'], 'log')
        self._configure_logging(settings, level='DEBUG')

    def apply_testing(self, settings):
        self._configure_logging(settings, level='DEBUG')

    def apply_stress(self, settings):
        self._configure_logging(settings)

    def apply_production(self, settings):
        self._configure_logging(settings)

if os.getenv('YANDEX_ENVIRONMENT_TYPE') == 'testing':
    YANDEX_DATA_CENTER = 'XXX'

Configuration.configurator_classes.extend((
    LoggingConfigurator, LocaleConfigurator, TouchHostConfigurator, MordaHostConfigurator,
    TrainsHostConfigurator, TravelHostConfigurator
))
Configuration().apply(globals())
