# coding: utf-8
from __future__ import absolute_import, division, print_function, unicode_literals

import itertools
from collections import defaultdict
from datetime import timedelta

from django.utils import translation

from rest_framework.decorators import api_view
from rest_framework.response import Response

from common.utils.date import daterange
from travel.library.python.tracing.instrumentation import child_span
from travel.rasp.wizards.train_wizard_api.lib.pgaas_price_store.tariff_direction_info_provider import tariff_direction_info_provider
from travel.rasp.wizards.train_wizard_api.serialization.direction_places import dump_query, load_query


def _get_iso_date(iso_datetime):
    return iso_datetime.split('T', 1)[0]


def _aggregate_places(infos):
    places_by_coach_type = defaultdict(list)
    for info in infos:
        for places_info in info.places:
            places_by_coach_type[places_info.coach_type].append(places_info)

    return [
        {
            'coach_type': coach_type,
            'count': sum(place.count for place in places),
            'max_seats_in_the_same_car': max(place.max_seats_in_the_same_car for place in places),
            'minimum_price': {
                'value': min(float(place.price.value) for place in places),
                'currency': places[0].price.currency  # TODO обрабатывать разные валюты
            }
        }
        for coach_type, places in places_by_coach_type.iteritems()
    ]


@api_view()
def direction_places_view(request):
    query = load_query(request.query_params)
    translation.activate(query.language)

    with child_span('train_wizard_api.views.direction_places.direction_places_view::make_departure_date_values'):
        departure_date_values = [
            departure_date.isoformat()
            for departure_date in daterange(query.min_departure_date,
                                            query.min_departure_date + timedelta(days=query.days))
        ]

    tariff_direction_infos, _updated_info = tariff_direction_info_provider.find(
        query.departure_point,
        query.arrival_point,
        query.min_departure_date,
        days=query.days,
    )

    with child_span('train_wizard_api.views.direction_places.direction_places_view::make_aggrefated_places'):
        aggregated_places = {
            departure_date: _aggregate_places(date_infos)
            for departure_date, date_infos in itertools.groupby(tariff_direction_infos,
                                                                key=lambda info: _get_iso_date(info.departure_dt))
        }

    with child_span('train_wizard_api.views.direction_places.direction_places_view::make_Response'):
        return Response(dict(dump_query(query), **{
            'groups': [
                {'departure_date': departure_date, 'places': aggregated_places.get(departure_date, [])}
                for departure_date in departure_date_values
            ]
        }))
