# coding: utf-8
from __future__ import absolute_import, division, print_function, unicode_literals

import ujson
from marshmallow import Schema, fields
from rest_framework.views import APIView
from rest_framework.response import Response

from travel.library.python.tracing.instrumentation import child_span
from travel.rasp.wizards.train_wizard_api.lib.pgaas_price_store.train_info_saver import TrainInfoSaver, train_info_saver  # noqa: UnusedImport
from travel.rasp.wizards.train_wizard_api.lib.train_facility_fabric import TrainFacilityFabric, train_facility_fabric  # noqa: UnusedImport
from travel.rasp.wizards.train_wizard_api.lib.train_info_parser import TrainInfoParser, train_info_parser  # noqa: UnusedImport
from travel.rasp.wizards.train_wizard_api.lib.train_tariff_fabric import train_tariff_fabric, TrainTariffFabric  # noqa: UnusedImport


class IndexTrainQuerySchema(Schema):
    departure_point_express_id = fields.Integer(required=True)
    arrival_point_express_id = fields.Integer(required=True)
    departure_at = fields.DateTime(required=True, load_from='departure_dt')
    number = fields.String(required=True)


class IndexTrainView(APIView):
    train_info_parser = None
    train_tariff_fabric = None
    train_info_saver = None
    train_facility_fabric = None

    def __init__(self, train_info_parser, train_tariff_fabric, train_facility_fabric, train_info_saver):
        # type: (TrainInfoParser, TrainTariffFabric, TrainFacilityFabric, TrainInfoSaver) -> None
        super(IndexTrainView, self).__init__()
        self._train_info_parser = train_info_parser
        self._train_tariff_fabric = train_tariff_fabric
        self._train_facility_fabric = train_facility_fabric
        self._train_tariff_info_saver = train_info_saver

    def post(self, request):
        with child_span('train_wizard_api.views.index_train.post::IndexTrainQuerySchema.load'):
            query, errors = IndexTrainQuerySchema().load(request.query_params)
        if errors:
            return Response(errors, status=400)

        with child_span('train_wizard_api.views.index_train.post::ujson.loads'):
            data = ujson.loads(request.body)

        electronic_ticket = data.get('electronic_ticket')
        with child_span('train_wizard_api.views.index_train.post::_train_info_parser.parse'):
            places_info = self._train_info_parser.parse(data['coaches'])
        with child_span('train_wizard_api.views.index_train.post::_train_tariff_fabric.make_tariffs_info'):
            tariffs_info = self._train_tariff_fabric.make_tariffs_info(places_info)
        with child_span('train_wizard_api.views.index_train.post::_train_facility_fabric.get_facilities_ids'):
            facilities_ids = self._train_facility_fabric.get_facilities_ids(places_info)

        self._train_tariff_info_saver.save(
            query=query,
            tariffs_info=tariffs_info,
            places_info=places_info,
            facilities_ids=facilities_ids,
            electronic_ticket=electronic_ticket
        )
        return Response('ok')


index_train_view = IndexTrainView.as_view(
    train_info_parser=train_info_parser,
    train_tariff_fabric=train_tariff_fabric,
    train_info_saver=train_info_saver,
    train_facility_fabric=train_facility_fabric,
)
