# coding: utf-8
from __future__ import absolute_import, division, print_function, unicode_literals

from django.utils import translation
from rest_framework import status
from rest_framework.decorators import api_view
from rest_framework.response import Response

from travel.rasp.wizards.train_wizard_api.direction.logger import legacy_direction_logger
from travel.rasp.wizards.train_wizard_api.direction.schedule_cache import schedule_cache
from travel.rasp.wizards.train_wizard_api.direction.segments import join_variants, make_segments, make_tomorrow_query, split_segments
from travel.rasp.wizards.train_wizard_api.lib.pgaas_price_store.tariff_direction_info_provider import tariff_direction_info_provider
from travel.rasp.wizards.train_wizard_api.lib.pgaas_price_store.train_info_provider import train_info_provider
from travel.rasp.wizards.train_wizard_api.lib.url_factories import host_provider, search_url_factory
from travel.rasp.wizards.train_wizard_api.serialization.legacy_direction import dump_segments, load_query
from travel.rasp.wizards.wizard_lib.direction.segments import find_schedule_segments
from travel.rasp.wizards.wizard_lib.event_date_query import make_event_date_query
from travel.rasp.wizards.wizard_lib.serialization.legacy_direction import dump_query
from travel.rasp.wizards.wizard_lib.utils.functional import compose
from travel.rasp.wizards.wizard_lib.utils.text import render_texts
from travel.rasp.wizards.wizard_lib.views.helpers.error_handlers import log_validation_error


def _get_search_urls(departure_point, arrival_point, when, tld, empty_variants):
    params = dict(
        from_point=departure_point,
        to_point=arrival_point,
        when=when,
        tld=tld,
        transport_type='train',
        empty_variants=empty_variants,
    )

    return {
        'url': search_url_factory.format_url(is_mobile=False, **params),
        'touch_url': search_url_factory.format_url(is_mobile=True, **params),
    }


def _get_texts(departure_point, arrival_point, departure_date, experiments):
    context = {
        'departure_point': departure_point,
        'arrival_point': arrival_point,
        'departure_date': departure_date,
    }
    return render_texts('direction/texts', context, experiments)


@api_view()
@log_validation_error
@legacy_direction_logger.decorate
def legacy_direction_view(request, log_context):
    query = load_query(request.query_params)
    translation.activate(query.language)
    log_context.store_query(query)

    departure_date_query = (
        make_event_date_query(query.departure_date, query.departure_point.pytz) if query.departure_date is not None else
        make_tomorrow_query(query.departure_point.pytz)
    )
    found_departure_date, raw_segments = find_schedule_segments(
        schedule_cache, query.departure_point, query.arrival_point, departure_date_query
    )
    if not raw_segments:
        return Response(status=status.HTTP_204_NO_CONTENT)

    tariff_direction_info = tariff_direction_info_provider.find(
        query.departure_point, query.arrival_point, found_departure_date
    )

    trains_info = train_info_provider.find(
        query.departure_point, query.arrival_point, found_departure_date
    )

    segments = make_segments(raw_segments, tariff_direction_info, trains_info)
    log_context.store_segments(segments)

    segments = compose(split_segments, query.filters.apply, query.sorting.apply, join_variants)(segments)
    log_context.store_filtered_segments(segments)

    if not segments:
        return Response(status=status.HTTP_204_NO_CONTENT)

    search_urls = _get_search_urls(
        query.departure_point,
        query.arrival_point,
        found_departure_date,
        query.tld,
        not segments,
    )

    response = {
        'default_transport': dict(
            dump_segments(
                segments=segments,
                departure_point=query.departure_point,
                arrival_point=query.arrival_point,
                when=found_departure_date,
                language=query.language,
                tld=query.tld,
                experiment_flags=query.experiment_flags
            ),
            **{'departure_date': found_departure_date.strftime('%Y-%m-%d')}
        ),
        'path_items': [dict(search_urls, **{'text': host_provider.get_host(
            is_mobile=False,
            transport_type='train',
            tld=query.tld,
        )})],
        'query': dump_query(query),
        'type': 'transports_with_default',
    }
    response.update(search_urls)
    response.update(_get_texts(
        query.departure_point,
        query.arrival_point,
        query.departure_date,
        query.experiment_flags,
    ))
    response['train_title'] = response['title']

    return Response(response)
