# coding: utf-8
from __future__ import absolute_import, division, print_function, unicode_literals

from itertools import izip
import logging
import urlparse

from django.conf import settings
from django.utils import translation
from opentracing import global_tracer
from rest_framework import status
from rest_framework.decorators import api_view
from rest_framework.response import Response

from common.utils.date import MSK_TZ
from travel.library.python.tracing.instrumentation import child_span, traced_function
from travel.rasp.wizards.train_wizard_api.direction.logger import open_direction_logger
from travel.rasp.wizards.train_wizard_api.direction.promotion import BRAND_PROMOTION, DURATION_PROMOTION, NULL_PROMOTION
from travel.rasp.wizards.train_wizard_api.direction.schedule_cache import schedule_cache
from travel.rasp.wizards.train_wizard_api.direction.segments import join_variants, make_segments, make_tomorrow_query, split_segments
from travel.rasp.wizards.train_wizard_api.lib.first_equal_predicate import FirstEqualPredicate
from travel.rasp.wizards.train_wizard_api.lib.pgaas_price_store.tariff_direction_info_provider import tariff_direction_info_provider
from travel.rasp.wizards.train_wizard_api.lib.pgaas_price_store.tariff_direction_info_saver import tariff_direction_info_saver
from travel.rasp.wizards.train_wizard_api.lib.pgaas_price_store.train_info_provider import train_info_provider
from travel.rasp.wizards.train_wizard_api.lib.storage_timed_execute import execute_all_with_timeout
from travel.rasp.wizards.train_wizard_api.lib.url_factories import host_provider, order_url_factory, search_url_factory
from travel.rasp.wizards.train_wizard_api.serialization.open_direction import dump_segment, load_query
from travel.rasp.wizards.train_wizard_api.tools.fake_segment_fabric import segment_fabric
from travel.rasp.wizards.wizard_lib.direction.segments import find_schedule_segments
from travel.rasp.wizards.wizard_lib.event_date_query import make_event_date_query
from travel.rasp.wizards.wizard_lib.experiment_flags import ExperimentFlag
from travel.rasp.wizards.wizard_lib.serialization.date import parse_date
from travel.rasp.wizards.wizard_lib.serialization.direction import dump_query
from travel.rasp.wizards.wizard_lib.serialization.point import parse_settlement_geoid
from travel.rasp.wizards.wizard_lib.serialization.price import dump_price
from travel.rasp.wizards.wizard_lib.serialization.segments import dump_duration, get_minimum_duration
from travel.rasp.wizards.wizard_lib.utils.functional import compose
from travel.rasp.wizards.wizard_lib.utils.text import render_texts
from travel.rasp.wizards.wizard_lib.utils.yasm_metric_manager import YasmMetricManager
from travel.rasp.wizards.wizard_lib.views.helpers.error_handlers import log_validation_error

logger = logging.getLogger(__name__)
yasm_metric_manager = YasmMetricManager(__name__, logger)


def _get_minimum_price(prices):
    prices = filter(None, prices)
    if prices:
        return min(prices)


def _dump_search_urls(query, found_departure_date, empty_variants=False):
    extra_params = query.filters and query.filters.get_search_params() or ()

    desktop_url = search_url_factory.format_url(
        is_mobile=False,
        from_point=query.departure_point,
        to_point=query.arrival_point,
        when=found_departure_date,
        tld=query.tld,
        transport_type='train',
        empty_variants=empty_variants,
        extra_params=extra_params,
    )
    mobile_url = search_url_factory.format_url(
        is_mobile=True,
        from_point=query.departure_point,
        to_point=query.arrival_point,
        when=found_departure_date,
        tld=query.tld,
        transport_type='train',
        empty_variants=empty_variants,
        extra_params=extra_params,
    )
    return desktop_url, mobile_url


def _dump_service_urls(query):
    desktop_host = host_provider.get_host(is_mobile=False, transport_type='train', tld=query.tld)
    mobile_host = host_provider.get_host(is_mobile=True, transport_type='train', tld=query.tld)
    desktop_url = urlparse.SplitResult(scheme='https', netloc=desktop_host, path='', query=None, fragment=None).geturl()
    mobile_url = urlparse.SplitResult(scheme='https', netloc=mobile_host, path='', query=None, fragment=None).geturl()
    return desktop_url, mobile_url


@traced_function
def _get_texts(departure_point, arrival_point, departure_date, is_brand, brand_title, experiments):
    context = {
        'departure_point': departure_point,
        'arrival_point': arrival_point,
        'departure_date': departure_date,
        'brand_title': brand_title,
        'is_brand': is_brand,
    }
    return render_texts('direction/texts', context, experiments)


def _get_snippet_for_radius_search(query):
    if query.original_departure_point is None and query.original_arrival_point is None:
        return {}

    context = {
        'departure_point': query.departure_point,
        'arrival_point': query.arrival_point,
        'original_departure_point': query.original_departure_point,
        'original_arrival_point': query.original_arrival_point,
    }
    snippet = render_texts('direction/texts_radius', context, query.experiment_flags)
    return {'nearest_points_snippet': snippet['title']}


def _get_promotion(query):
    experiment_flags = query.experiment_flags
    return (
        BRAND_PROMOTION if ExperimentFlag.BEST_OFFERS_BRAND in experiment_flags else
        DURATION_PROMOTION if ExperimentFlag.BEST_OFFERS_DURATION in experiment_flags else
        NULL_PROMOTION
    )


def _get_minimum_duration_price_and_minimum_prices(segments):
    minimum_prices = tuple(
        min(places_group.price for places_group in segment.places) if segment.places else None
        for segment in segments
    )
    minimum_price = _get_minimum_price(minimum_prices)
    minimum_duration = get_minimum_duration(segments) if segments else None
    return minimum_duration, minimum_price, minimum_prices


@api_view()
@log_validation_error
def _open_direction_view(request, log_context):
    yasm_metric_manager.send_one('request_cnt')

    query = load_query(request.query_params)

    if query.departure_point is None or query.arrival_point is None:
        with child_span('train_wizard_api.views.direction._open_direction_view::return_response_for_pointless_query'):
            departure_point = (
                parse_settlement_geoid(
                    query_params=request.query_params,
                    settlement_geoid_name='geo_id',
                )
                if query.departure_point is None and request.query_params.get('geo_id') is not None
                else query.departure_point
            )

            query = query._replace(departure_point=departure_point)

            departure_date = (
                parse_date(
                    value=request.query_params.get('departure_date'),
                    local_tz=departure_point.pytz if departure_point else MSK_TZ,
                    today_default=True,
                    ignore_past=False,
                )
                if query.departure_date is None
                else query.departure_date
            )

            response = {
                'found_departure_date': departure_date.strftime('%Y-%m-%d') if departure_date else None,
                'path_items': [
                    {
                        'text': 'travel.yandex.ru/trains/',
                        'touch_url': 'https://travel.yandex.ru/trains/',
                        'url': 'https://travel.yandex.ru/trains/',
                    }
                ],
            }
            response.update(dump_query(query, is_filters_needed=False))

            is_brand, brand_title = query.filters.get_brand_filter().get_brand_title()

            response.update(_get_texts(
                departure_point=departure_point,
                arrival_point=query.arrival_point,
                departure_date=departure_date,
                brand_title=brand_title,
                is_brand=is_brand,
                experiments=query.experiment_flags,
            ))

            yasm_metric_manager.send_one('pointless_cnt')

            return Response(response)

    promotion = _get_promotion(query)
    translation.activate(query.language)
    log_context.store_query(query)

    departure_date_query = (
        make_event_date_query(query.departure_date, query.departure_point.pytz)
        if query.departure_date is not None else
        make_tomorrow_query(query.departure_point.pytz)
    )
    found_departure_date, raw_segments = find_schedule_segments(
        schedule_cache, query.departure_point, query.arrival_point, departure_date_query
    )
    if not raw_segments:
        yasm_metric_manager.send_one('no_data_cnt')
        return Response(status=status.HTTP_204_NO_CONTENT, headers={'Content-Length': 0})

    provider_parameters = [
        (provider, (query.departure_point, query.arrival_point, found_departure_date))
        for provider in (tariff_direction_info_provider, train_info_provider)
    ]
    tariff_direction_info, trains_info = execute_all_with_timeout(
        provider_parameters,
        settings.DBAAS_TRAIN_WIZARD_API_SELECT_TIMEOUT
    )

    segments = make_segments(raw_segments, tariff_direction_info, trains_info)
    log_context.store_segments(segments)

    all_minimum_duration, all_minimum_price, _ = _get_minimum_duration_price_and_minimum_prices(segments)

    mark_the_cheapest_train, mark_the_fastest_train = (
        (lambda v: False, lambda v: False) if len(segments) < 2 else
        (FirstEqualPredicate(all_minimum_price), FirstEqualPredicate(all_minimum_duration))
    )
    if all_minimum_price is None:
        def mark_the_cheapest_train(value):
            return False

    segments = compose(
        split_segments, query.filters.apply, query.sorting.apply, join_variants, promotion.apply
    )(segments)
    log_context.store_filtered_segments(segments)

    minimum_duration, minimum_price, minimum_prices = _get_minimum_duration_price_and_minimum_prices(segments)

    if settings.DEV_GENERATE_PRICES:
        price_segments = segment_fabric.make(query.departure_point, query.arrival_point,
                                             found_departure_date, segments)
        tariff_direction_info_saver.save(price_segments)

    desktop_service_url, mobile_service_url = _dump_service_urls(query)
    desktop_search_url, mobile_search_url = _dump_search_urls(
        query,
        found_departure_date,
        empty_variants=not raw_segments
    )

    mobile_order_url_context = order_url_factory.build_context(
        is_mobile=True,
        from_point=query.departure_point,
        to_point=query.arrival_point,
        when=found_departure_date,
        tld=query.tld,
        transport_type='train',
    )

    desktop_order_url_context = order_url_factory.build_context(
        is_mobile=False,
        from_point=query.departure_point,
        to_point=query.arrival_point,
        when=found_departure_date,
        tld=query.tld,
        transport_type='train',
    )

    with child_span('train_wizard_api.views.direction._open_direction_view::form_response'):
        response = {
            'found_departure_date': found_departure_date.strftime('%Y-%m-%d'),
            'minimum_price': dump_price(minimum_price),
            'minimum_duration': dump_duration(minimum_duration),
            'path_items': [
                {
                    'text': host_provider.get_host(
                        is_mobile=False,
                        transport_type='train',
                        tld=query.tld,
                    ),
                    'touch_url': mobile_service_url,
                    'url': desktop_service_url,
                },
                {
                    'text': translation.ugettext('train_tickets_from_departure_point_to_arrival_point') % {
                        'departure': query.departure_point.L_title(case='genitive'),
                        'arrival': query.arrival_point.L_title(case='accusative'),
                    },
                    'touch_url': mobile_search_url,
                    'url': desktop_search_url,
                }
            ],
            'search_touch_url': mobile_search_url,
            'search_url': desktop_search_url,
            'segments': [
                dict(
                    dump_segment(segment, query.language),
                    **{
                        'minimum_price': dump_price(min_price),
                        'order_touch_url': mobile_order_url_context.add_segment_info(
                            segment.train_number, segment.departure_local_dt, segment.provider
                        ).format_url(),
                        'order_url': desktop_order_url_context.add_segment_info(
                            segment.train_number, segment.departure_local_dt, segment.provider
                        ).format_url(),
                        'is_the_fastest': mark_the_fastest_train(segment.duration),
                        'is_the_cheapest': mark_the_cheapest_train(min_price),
                    }
                )
                for segment, min_price in izip(segments, minimum_prices)
            ],
            'total': len(segments),
        }
    response.update(dump_query(query))
    response.update(_get_snippet_for_radius_search(query))

    is_brand, brand_title = query.filters.get_brand_filter().get_brand_title()

    response.update(_get_texts(
        departure_point=query.departure_point,
        arrival_point=query.arrival_point,
        departure_date=query.departure_date,
        brand_title=brand_title,
        is_brand=is_brand,
        experiments=query.experiment_flags,
    ))
    yasm_metric_manager.send_one('with_data_cnt')
    return Response(response)


open_direction_view = open_direction_logger.decorate(_open_direction_view)
