# coding: utf-8
from __future__ import absolute_import, division, print_function, unicode_literals

import json
import logging
import requests

from django.conf import settings
from marshmallow import fields, Schema, validate
from marshmallow.decorators import post_load
from rest_framework.decorators import api_view
from rest_framework.response import Response

from common.settings.configuration import Configuration
from common.settings.utils import define_setting
from travel.rasp.library.python.common23.date.environment import now_utc
from travel.library.python.tracing.instrumentation import child_span
from travel.rasp.wizards.train_wizard_api.lib.pgaas_price_store.tariff_direction_info_provider import tariff_direction_info_provider
from travel.rasp.wizards.train_wizard_api.lib.pgaas_price_store.tariff_direction_info_saver import tariff_direction_info_saver

log = logging.getLogger(__name__)

define_setting(
    'SEARCH_API_URL',
    default='http://testing.search-api.trains.internal.yandex.ru/',
    env={
        Configuration.PRODUCTION: 'http://production.search-api.trains.internal.yandex.ru/',
        Configuration.TESTING: 'http://testing.search-api.trains.internal.yandex.ru/',
        Configuration.STRESS: 'http://load.search-api.trains.internal.yandex.ru/'
    }
)

define_setting(
    'SEARCH_API_INDEXER_TIMEOUT', default=5,
    env={Configuration.PRODUCTION: 2, Configuration.STRESS: 2}
)

define_setting('ENABLE_PROXY_INDEXER_TO_SEARCH_API', default=True, converter=lambda x: bool(x))


class DirectionTariffInfoSchema(Schema):
    departure_point_express_id = fields.Integer(required=True)
    arrival_point_express_id = fields.Integer(required=True)
    departure_date = fields.Date(required=True)
    data = fields.List(fields.Dict(), required=True)

    @post_load()
    def to_dict(self, data):
        today = now_utc()

        return {
            'departure_point_express_id': data['departure_point_express_id'],
            'arrival_point_express_id': data['arrival_point_express_id'],
            'departure_date': data['departure_date'],
            'created_at': today,
            'updated_at': today,
            'data': data['data'],
        }


@api_view(['POST'])
def index_view(request):
    raw_params = request.query_params.copy()
    try:
        raw_params['data'] = json.loads(request.body)
    except ValueError:
        return Response('Error: incorrect json', status=400)

    with child_span('train_wizard_api.views.prices.index_view::DirectionTariffInfoSchema.load'):
        result = DirectionTariffInfoSchema().load(raw_params.dict())
    if result.errors:
        return Response(result.errors, status=400)

    tariff_direction_info_saver.save(result.data)

    # proxy request to search-api indexer
    if settings.ENABLE_PROXY_INDEXER_TO_SEARCH_API:
        try:
            proxy_response = requests.post(
                '{}{}'.format(settings.SEARCH_API_URL, 'indexer/api/direction/'),
                params=request.query_params,
                headers={
                    'Content-Type': 'application/json',
                },
                data=request.body,
                timeout=settings.SEARCH_API_INDEXER_TIMEOUT)
        except (requests.ConnectionError, requests.Timeout) as ex:
            log.exception('SEARCH_API indexer request error. {}'.format(repr(ex)))
            raise

        try:
            proxy_response.raise_for_status()
        except requests.HTTPError:
            log.exception('SEARCH_API indexer response bad status. {}'.format(proxy_response.request.body))
            raise

    return Response('ok')


class DirectionTariffSearchInfoSchema(Schema):
    departure_point_key = fields.String(required=True, validate=validate.Regexp('^[sc]\d+$'))
    arrival_point_key = fields.String(required=True, validate=validate.Regexp('^[sc]\d+$'))
    departure_date = fields.Date(required=True)


@api_view()
def search_view(request):
    result = DirectionTariffSearchInfoSchema().load(request.query_params.copy())
    if result.errors:
        return Response(result.errors, status=400)
    query = result.data

    return Response(tariff_direction_info_provider.find(
        query['departure_point_key'],
        query['arrival_point_key'],
        query['departure_date']
    ))
