# coding: utf-8
from __future__ import absolute_import, division, print_function, unicode_literals

import bisect
import itertools
from datetime import time

from rest_framework import serializers


class BaseFilter(object):
    _variant_values = None

    def __init__(self, selected):
        self._selected = selected
        self._available = set()

    @property
    def is_bound(self):
        return self._variant_values is not None

    def bind(self, variants):
        raise NotImplementedError

    def list_selectors(self):
        raise NotImplementedError

    def update_availability(self, selectors):
        assert self.is_bound
        self._available.update(itertools.compress(self._variant_values, selectors))

    def selected_value(self):
        return self._selected

    def get_search_params(self):
        return ()


class BaseTimeFilter(BaseFilter):
    VALUES, BOUNDS = zip(('00:00-06:00', time.min),
                         ('06:00-12:00', time(6)),
                         ('12:00-18:00', time(12)),
                         ('18:00-24:00', time(18)))
    time_getter = None

    def __init__(self, selected):
        self._selected = selected
        super(BaseTimeFilter, self).__init__(selected)

    @classmethod
    def load(cls, values):
        selected = set()
        for value in values:
            if value not in cls.VALUES:
                raise serializers.ValidationError('invalid time filter value: it should be one of [{}]'
                                                  .format(', '.join(cls.VALUES)))
            selected.add(value)
        return cls(selected)

    def bind(self, variants):
        self._variant_values = [
            self.VALUES[bisect.bisect(self.BOUNDS, self._get_time(v.segment)) - 1]
            for v in variants
        ]

    def list_selectors(self):
        assert self.is_bound
        selected, variant_values = self._selected, self._variant_values

        if not selected:
            return [True]*len(variant_values)

        return [variant_value in selected for variant_value in variant_values]

    def dump(self):
        selected, available = self._selected, self._available
        return [
            {
                'value': range_value,
                'selected': range_value in selected,
                'available': range_value in available
            }
            for range_value in self.VALUES
        ]

    def _get_time(self, segment):
        raise NotImplementedError


class DepartureTimeFilter(BaseTimeFilter):
    def _get_time(self, segment):
        return segment.departure_local_dt.time()


class ArrivalTimeFilter(BaseTimeFilter):
    def _get_time(self, segment):
        return segment.arrival_local_dt.time()


class BaseFilters(object):
    FACTORIES = ()

    def __init__(self, filters):
        self._filters = filters

    @classmethod
    def load(cls, query_params):
        return cls(tuple(filter_factory(filter(None, query_params.getlist(filter_name)))
                         for filter_name, filter_factory in cls.FACTORIES))

    def apply(self, variants):
        filters_selectors = []

        for filter_obj in self._filters:
            filter_obj.bind(variants)
            filters_selectors.append(filter_obj.list_selectors())

        for i, filter_obj in enumerate(self._filters):
            filter_obj.update_availability(map(all, itertools.izip(*(filters_selectors[:i] +
                                                                     filters_selectors[i + 1:]))))

        return tuple(itertools.compress(variants, itertools.imap(all, itertools.izip(*filters_selectors))))

    def dump(self):
        return {
            filter_name: filter_obj.dump()
            for (filter_name, _filter_factory), filter_obj in itertools.izip(self.FACTORIES, self._filters)
        }

    def get_search_params(self):
        return sum((filter_obj.get_search_params() for filter_obj in self._filters), ())

    def selected_values_by_name(self):
        return {
            filter_name: filter_obj.selected_value()
            for (filter_name, _filter_factory), filter_obj in itertools.izip(self.FACTORIES, self._filters)
        }
