# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import calendar
import collections
import time
from logging import getLogger

from common.models.geo import Settlement, Station


def dump_datetime(dt):
    if dt is None:
        return None
    return calendar.timegm(dt.timetuple())


def typify_point(point):
    if point is None:
        return None, None
    if isinstance(point, Settlement):
        return point.id, None
    if isinstance(point, Station):
        return None, point.id
    raise RuntimeError('unknown type')


def default_query_adapter(query):
    departure_settlement_id, departure_station_id = typify_point(query.departure_point)
    arrival_settlement_id, arrival_station_id = typify_point(query.arrival_point)

    query_entry = {
        'departure_settlement_id': departure_settlement_id,
        'departure_station_id': departure_station_id,
        'arrival_settlement_id': arrival_settlement_id,
        'arrival_station_id': arrival_station_id,
        'departure_date': dump_datetime(query.departure_date),
        'language': query.language,
        'experiment_flags': [flag.value for flag in query.experiment_flags],
        'sorting': query.sorting.value
    }

    for name, values in query.filters.selected_values_by_name().items():
        if not values:
            continue

        if isinstance(values, collections.Iterable):
            query_entry['filters_{}'.format(name)] = list(sorted(values))
        else:
            query_entry['filters_{}'.format(name)] = str(values)

    return query_entry


def default_segments_adapter(segments):
    return {'all_segments_count': len(segments)}


class DirectionLoggerContext(object):
    def __init__(self, data, query_adapter, segments_adapter):
        self._data = data
        self._query_adapter = query_adapter
        self._segments_adapter = segments_adapter

    def store_query(self, query):
        self._data.update(self._query_adapter(query))

    def _store_segments(self, segments, prefix):
        self._data.update({
            '{}_{}'.format(prefix, entry_name): entry_value
            for entry_name, entry_value in self._segments_adapter(segments).iteritems()
        })

    def store_segments(self, segments):
        self._store_segments(segments, 'before_filters')

    def store_filtered_segments(self, segments):
        self._store_segments(segments, 'filters')

    def get_data(self):
        return self._data


class DirectionLogger(object):
    def __init__(
        self,
        time_provider=time,
        yt_logger=getLogger('yt.direction'),
        query_adapter=default_query_adapter,
        segments_adapter=default_segments_adapter,
        defaults=None,
    ):
        self._time_provider = time_provider
        self._yt_logger = yt_logger
        self._defaults = defaults or {}
        self._query_adapter = query_adapter
        self._segments_adapter = segments_adapter

    def start_log(self):
        return DirectionLoggerContext(
            dict(self._defaults, **{'unixtime': int(self._time_provider.time())}),
            self._query_adapter,
            self._segments_adapter,
        )

    def decorate(self, func):
        def _wrapper(*args, **kwargs):
            log_context = self.start_log()
            result = func(log_context=log_context, *args, **kwargs)
            self._yt_logger.info(log_context.get_data())
            return result

        _wrapper.__name__ = func.__name__
        return _wrapper
