# coding: utf8
from __future__ import absolute_import, division, print_function, unicode_literals

from contextlib2 import contextmanager
from geoindex import GeoGridIndex, GeoPoint

from common.models.geo import Settlement
from travel.rasp.wizards.wizard_lib.geobase_region_type import GeobaseRegionType, LikeSettlementTypesIds


class OptimalDirectionProvider(object):
    def __init__(self, geobase, schedule_cache, logger):
        # type: (Geobase, any, Logger) -> None
        self._settlement_reachable_cache = None
        self._geo_index = None

        self._geobase = geobase
        self._schedule_cache = schedule_cache
        self.logger = logger

    @contextmanager
    def using_precache(self):
        if not self.is_precached():
            self.build_cache()
            try:
                yield
            finally:
                self.clean()
        else:
            yield

    def is_precached(self):
        return self._geo_index is not None

    def clean(self):
        self._settlement_reachable_cache = None
        self._geo_index = None

    def build_cache(self):
        settlement_id_to_model = {}
        settlement_reachable_cache = self._schedule_cache.get_settlement_reachable_cache()
        for from_settlement_id, to_settlement_ids in settlement_reachable_cache.items():
            for settlement_id in list(to_settlement_ids) + [from_settlement_id]:
                if settlement_id not in settlement_id_to_model:
                    settlement_id_to_model[settlement_id] = Settlement.objects.get(id=settlement_id)
        geo_index = GeoGridIndex(precision=2)
        for settlement in settlement_id_to_model.values():
            if settlement.latitude is not None and settlement.longitude is not None:
                geo_index.add_point(GeoPoint(settlement.latitude, settlement.longitude, ref=settlement))

        self._settlement_reachable_cache = settlement_reachable_cache
        self._geo_index = geo_index

    def find_optimal_direction(self, departure_settlement_geo_id, arrival_settlement_geo_id, radius):
        # type: (int, int) -> Tuple[Optional[Settlement], Optional[Settlement]]
        departure_geo_object = self._find_geo_object(departure_settlement_geo_id)
        arrival_geo_object = self._find_geo_object(arrival_settlement_geo_id)

        departure_country_id = (
            self._geobase.find_country_id(departure_settlement_geo_id, national_version='ru')
            if departure_geo_object
            else None
        )
        arrival_country_id = (
            self._geobase.find_country_id(arrival_settlement_geo_id, national_version='ru')
            if arrival_geo_object
            else None
        )

        if not self._can_find_optiomal_direction(departure_geo_object, arrival_geo_object):
            return None, None

        nearest_departure_settlements = self._find_nearest_settlements_with_distance(
            departure_geo_object, radius, departure_country_id
        )
        nearest_arrival_settlements = self._find_nearest_settlements_with_distance(
            arrival_geo_object, radius, arrival_country_id
        )

        return self._find_optional_direction(
            nearest_departure_settlements, nearest_arrival_settlements
        )

    def _can_find_optiomal_direction(self, departure, arrival):
        if departure is None:
            self.logger.debug("Can not find optimal direction, because departure point is None.")
            return False
        if arrival is None:
            self.logger.debug("Can not find optimal direction, because arrival point is None.")
            return False

        return True

    def _find_geo_object(self, geo_id):
        try:
            geo_object = self._geobase.region_by_id(geo_id)
        except RuntimeError:
            self.logger.debug("Geo object does not exist. geo_id: %d", geo_id)
            return None

        try:
            geo_object_type = GeobaseRegionType(geo_object.type)
        except ValueError:
            self.logger.debug("Geo object has unknown type. geo_id: %d type_id: %d", geo_object.id, geo_object.type)
            return None

        if geo_object_type in LikeSettlementTypesIds:
            return geo_object
        else:
            self.logger.debug("Geo object has wrong type. geo_id: %d type: %s", geo_id, geo_object_type)
        return None

    def _find_nearest_settlements_with_distance(self, geo_object, radius, country_id=None):
        result = []
        for point, distance in self._geo_index.get_nearest_points(
                GeoPoint(geo_object.latitude, geo_object.longitude), radius, 'km'):
            if country_id is not None and point.ref.country_id == country_id:
                result.append((point.ref, distance))
        return result

    def _find_optional_direction(self, nearest_departure_settlements, nearest_arrival_settlements):
        optional_direction = None
        for departure_settlement, departure_distance in nearest_departure_settlements:
            for arrival_settlement, arrival_distance in nearest_arrival_settlements:
                if departure_settlement.id in self._settlement_reachable_cache:
                    ids = self._settlement_reachable_cache[departure_settlement.id]
                    if arrival_settlement.id in ids:
                        total_distance = departure_distance + arrival_distance
                        if optional_direction is None or total_distance < optional_direction[0]:
                            optional_direction = (total_distance, departure_settlement, arrival_settlement)
        if optional_direction is None:
            return None, None
        return optional_direction[1], optional_direction[2]
